<?php

namespace Drupal\glidejs\Plugin\views\style;

use Drupal\Core\Form\FormStateInterface;
use Drupal\views\Plugin\views\style\StylePluginBase;

/**
 * Style plugin to render a view using Glide.js.
 *
 * @ViewsStyle(
 *   id = "glide",
 *   title = @Translation("Glide Carousel"),
 *   help = @Translation("Displays rows in a Glide.js carousel."),
 *   theme = "views_view_style_glide",
 *   display_types = {"normal"}
 * )
 */
class GlideStyle extends StylePluginBase {

  /**
   * {@inheritdoc}
   */
  protected $usesRowPlugin = TRUE;

  /**
   * {@inheritdoc}
   */
  public function defineOptions() {
    $options = parent::defineOptions();

    $options += [
      'start_at' => ['default' => 0],
      'type' => ['default' => 'slider'],
      'per_view' => ['default' => 1],
      'focus_at' => ['default' => 0],
      'gap' => ['default' => 10],
      'autoplay' => ['default' => FALSE],
      'autoplay_speed' => ['default' => FALSE],
      'hover_pause' => ['default' => TRUE],
      'keyboard' => ['default' => TRUE],
      'bound' => ['default' => FALSE],
      'disable_swipe' => ['default' => FALSE],
      'swipe_threshold' => ['default' => 80],
      'disable_drag' => ['default' => FALSE],
      'drag_threshold' => ['default' => 120],
      'unlimited_touch' => ['default' => TRUE],
      'per_touch' => ['default' => FALSE],
      'touch_ratio' => ['default' => 0.5],
      'touch_angle' => ['default' => 45],
      'animation_duration' => ['default' => 400],
      'rewind' => ['default' => TRUE],
      'rewind_duration' => ['default' => 800],
      'animation_timing_func' => ['default' => 'cubic-bezier(0.680, -0.550, 0.265, 1.550)'],
      'direction' => ['default' => 'ltr'],
      'breakpoints' => ['default' => ''],
    ];

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    parent::buildOptionsForm($form, $form_state);

    // Type: slider or carousel.
    $form['type'] = [
      '#type' => 'select',
      '#title' => $this->t('Type'),
      '#default_value' => $this->options['type'],
      '#options' => [
        'carousel' => $this->t('Carousel'),
        'slider' => $this->t('Slider'),
      ],
      '#description' => $this->t('<p><strong>Type of movement.</strong></p><ul><li><strong>slider</strong> – @slider_desc</li><li><strong>carousel</strong> – @carousel_desc</li></ul>Default: @default.',
        [
          '@slider_desc' => 'rewinds to start/end at first or last slide.',
          '@carousel_desc' => 'changes slides continuously without rewinding.',
          '@default' => 'slider',
        ]
      ),
    ];

    // Focus position of active slide.
    $form['focus_at'] = [
      '#type' => 'textfield',
      '#title' => $this->t('FocusAt'),
      '#default_value' => $this->options['focus_at'],
      '#description' => $this->t(
        '<p><strong>Focus currently active slide</strong> at a specified position in the track.</p><ul><li><strong>center</strong> – @center_desc</li><li><strong>0, 1, 2...</strong> – @index_desc</li></ul>Default: @default.',
        [
          '@center_desc' => 'active slide is centered.',
          '@index_desc' => 'zero-based index of focus.',
          '@default' => '0',
        ]
      ),
    ];

    // Gap between slides.
    $form['gap'] = [
      '#type' => 'number',
      '#title' => $this->t('Gap'),
      '#default_value' => $this->options['gap'],
      '#min' => 0,
      '#description' => $this->t('Size of gap between slides. Default: 10.'),
    ];

    // Start at specific slide.
    $form['start_at'] = [
      '#type' => 'number',
      '#title' => $this->t('Start at'),
      '#default_value' => $this->options['start_at'],
      '#min' => 0,
      '#description' => $this->t(
        'Zero-based index of initial slide. Default: @start_at.', [
          '@start_at' => $this->options['start_at'],
        ]
      ),
    ];

    // Slides per view.
    $form['per_view'] = [
      '#type' => 'number',
      '#title' => $this->t('Slides per view'),
      '#default_value' => $this->options['per_view'],
      '#min' => 1,
      '#description' => $this->t(
        'Number of slides visible at once. Default: @per_view.', [
          '@per_view' => $this->options['per_view'],
        ]
      ),
    ];

    // Autoplay options.
    $form['autoplay'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Autoplay'),
      '#default_value' => $this->options['autoplay'],
      '#description' => $this->t('Automatically slide to the next item. Default: FALSE.'),
    ];

    $form['autoplay_speed'] = [
      '#type' => 'number',
      '#title' => $this->t('Autoplay speed (ms)'),
      '#default_value' => $this->options['autoplay_speed'],
      '#min' => 100,
      '#step' => 100,
      '#description' => $this->t('Time interval between automatic slide changes.'),
      '#states' => [
        'visible' => [
          ':input[name="style_options[autoplay]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Hover pause.
    $form['hover_pause'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Hover Pause'),
      '#default_value' => $this->options['hover_pause'],
      '#description' => $this->t('Pause autoplay on mouseover. Default: TRUE.'),
    ];

    // Keyboard navigation.
    $form['keyboard'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Keyboard'),
      '#default_value' => $this->options['keyboard'],
      '#description' => $this->t('Enable left/right arrow key navigation. Default: TRUE.'),
    ];

    // Bound slider.
    $form['bound'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Bound'),
      '#default_value' => $this->options['bound'],
      '#description' => $this->t(
        'Stops sliding past last slide. Works only with slider type and non-centered focusAt. Default: FALSE.'
      ),
    ];

    // Swipe/drag options.
    $form['disable_swipe'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Disable swipe'),
      '#default_value' => $this->options['disable_swipe'],
    ];

    $form['swipe_threshold'] = [
      '#type' => 'number',
      '#title' => $this->t('Swipe Threshold'),
      '#default_value' => $this->options['swipe_threshold'],
      '#min' => 0,
      '#description' => $this->t('Minimum swipe distance to change slide. Default: 80.'),
      '#states' => [
        'visible' => [
          ':input[name="style_options[disable_swipe]"]' => ['checked' => FALSE],
        ],
      ],
    ];

    $form['disable_drag'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Disable drag'),
      '#default_value' => $this->options['disable_drag'],
    ];

    $form['drag_threshold'] = [
      '#type' => 'number',
      '#title' => $this->t('Drag Threshold'),
      '#default_value' => $this->options['drag_threshold'],
      '#min' => 0,
      '#description' => $this->t('Minimum mouse drag distance to change slide. Default: 120.'),
      '#states' => [
        'visible' => [
          ':input[name="style_options[disable_drag]"]' => ['checked' => FALSE],
        ],
      ],
    ];

    $form['unlimited_touch'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Unlimited per touch'),
      '#default_value' => $this->options['unlimited_touch'],
      '#description' => $this->t('Allow any number of slides per swipe/drag. Default: TRUE.'),
    ];

    $form['per_touch'] = [
      '#type' => 'number',
      '#title' => $this->t('Per Touch'),
      '#default_value' => $this->options['per_touch'],
      '#min' => 1,
      '#description' => $this->t('Maximum slides per swipe/drag if Unlimited per touch is unchecked.'),
      '#states' => [
        'visible' => [
          ':input[name="style_options[unlimited_touch]"]' => ['checked' => FALSE],
        ],
      ],
    ];

    $form['touch_ratio'] = [
      '#type' => 'number',
      '#title' => $this->t('Touch Ratio'),
      '#default_value' => $this->options['touch_ratio'],
      '#min' => 0,
      '#max' => 1,
      '#step' => 0.05,
      '#description' => $this->t('Slide movement ratio for swipe/drag. Default: 0.5.'),
    ];

    $form['touch_angle'] = [
      '#type' => 'number',
      '#title' => $this->t('Touch Angle'),
      '#default_value' => $this->options['touch_angle'],
      '#min' => 0,
      '#max' => 360,
      '#step' => 1,
      '#description' => $this->t('Angle required to trigger slide movement. Default: 45.'),
    ];

    $form['animation_duration'] = [
      '#type' => 'number',
      '#title' => $this->t('Animation Duration'),
      '#default_value' => $this->options['animation_duration'],
      '#min' => 0,
      '#step' => 50,
      '#description' => $this->t('Animation duration in milliseconds. Default: 400.'),
    ];

    $form['rewind'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Rewind'),
      '#default_value' => $this->options['rewind'],
      '#description' => $this->t('Enable rewind to first/last slide. Works only with slider type. Default: TRUE.'),
      '#states' => [
        'visible' => [
          ':input[name="style_options[type]"]' => ['value' => 'slider'],
        ],
      ],
    ];

    $form['rewind_duration'] = [
      '#type' => 'number',
      '#title' => $this->t('Rewind Duration'),
      '#default_value' => $this->options['rewind_duration'],
      '#min' => 0,
      '#step' => 50,
      '#description' => $this->t('Rewind animation duration in milliseconds. Default: 800.'),
      '#states' => [
        'visible' => [
          ':input[name="style_options[type]"]' => ['value' => 'slider'],
        ],
      ],
    ];

    $form['animation_timing_func'] = [
      '#type' => 'select',
      '#title' => $this->t('Animation Timing Function'),
      '#default_value' => $this->options['animation_timing_func'],
      '#options' => [
        'cubic-bezier(0.680, -0.550, 0.265, 1.550)' => $this->t('bounce'),
        'linear' => $this->t('linear'),
        'ease' => $this->t('ease'),
        'ease-in' => $this->t('ease-in'),
        'ease-out' => $this->t('ease-out'),
        'ease-in-out' => $this->t('ease-in-out'),
      ],
      '#description' => $this->t('Easing function for animation. Default: bounce.'),
    ];

    $form['direction'] = [
      '#type' => 'select',
      '#title' => $this->t('Direction'),
      '#default_value' => $this->options['direction'],
      '#options' => [
        'ltr' => $this->t('Left to right (LTR)'),
        'rtl' => $this->t('Right to left (RTL)'),
      ],
      '#description' => $this->t('Moving direction. Default: ltr.'),
    ];

    // Breakpoints field for responsive settings.
    $form['breakpoints'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Breakpoints'),
      '#default_value' => $this->options['breakpoints'] ?? '',
      '#description' => $this->t(
        'JSON object for responsive settings.'
      ),
      '#element_validate' => [[get_class($this), 'validateJson']],
    ];
  }

  /**
   * Validate JSON field for breakpoints.
   */
  public static function validateJson($element, FormStateInterface $form_state) {
    $value = trim($element['#value']);
    if ($value !== '' && json_decode($value) === NULL) {
      $form_state->setError($element, t('The Breakpoints field must contain valid JSON.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function render() {
    $build = parent::render();

    $autoplay = $this->options['autoplay'] ? (int) $this->options['autoplay_speed'] : FALSE;
    $swipe_threshold = !empty($this->options['disable_swipe']) ? FALSE : (int) $this->options['swipe_threshold'];
    $drag_threshold = !empty($this->options['disable_drag']) ? FALSE : (int) $this->options['drag_threshold'];
    $per_touch = !empty($this->options['unlimited_touch']) ? FALSE : (int) $this->options['per_touch'];
    $breakpoints = !empty($this->options['breakpoints']) ? json_decode($this->options['breakpoints'], TRUE) ?? [] : [];

    $view_id = $this->view->id();
    $display_id = $this->view->current_display;

    $build['#attached']['library'][] = 'glidejs/glide_external';
    $build['#attached']['library'][] = 'glidejs/glide';

    $build['#attached']['drupalSettings']['glidejs'][$view_id][$display_id] = [
      'type' => $this->options['type'],
      'startAt' => $this->options['start_at'],
      'perView' => $this->options['per_view'],
      'focusAt' => $this->options['focus_at'],
      'gap' => $this->options['gap'],
      'autoplay' => $autoplay,
      'hoverpause' => $this->options['hover_pause'],
      'keyboard' => $this->options['keyboard'],
      'bound' => $this->options['bound'],
      'swipeThreshold' => $swipe_threshold,
      'dragThreshold' => $drag_threshold,
      'perTouch' => $per_touch,
      'touchRatio' => $this->options['touch_ratio'],
      'touchAngle' => $this->options['touch_angle'],
      'animationDuration' => $this->options['animation_duration'],
      'rewind' => $this->options['rewind'],
      'rewindDuration' => $this->options['rewind_duration'],
      'animationTimingFunc' => $this->options['animation_timing_func'],
      'direction' => $this->options['direction'],
      'breakpoints' => $breakpoints,
    ];

    $build['#attributes']['class'][] = 'js-glide-view';
    $build['#attributes']['data-glide-view-id'] = $view_id;
    $build['#attributes']['data-glide-display-id'] = $display_id;

    return $build;
  }

}
