<?php

namespace Drupal\glossify_taxonomy\Plugin\Filter;

use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Path\CurrentPathStack;
use Drupal\Core\Render\RendererInterface;
use Drupal\filter\FilterProcessResult;
use Drupal\glossify\GlossifyBase;
use Drupal\taxonomy\Entity\Vocabulary;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Filter to find and process found taxonomy terms in the fields value.
 *
 * @Filter(
 *   id = "glossify_taxonomy",
 *   title = @Translation("Glossify: Tooltips with taxonomy"),
 *   type = Drupal\filter\Plugin\FilterInterface::TYPE_TRANSFORM_IRREVERSIBLE,
 *   settings = {
 *     "glossify_taxonomy_case_sensitivity" = TRUE,
 *     "glossify_taxonomy_first_only" = TRUE,
 *     "glossify_taxonomy_ignore_tags" = "",
 *     "glossify_taxonomy_type" = "tooltips",
 *     "glossify_taxonomy_tooltip_truncate" = FALSE,
 *     "glossify_taxonomy_vocabs" = NULL,
 *     "glossify_taxonomy_urlpattern" = "/taxonomy/term/[id]",
 *     "glossify_taxonomy_synonyms_field" = "",
 *   },
 *   weight = -10
 * )
 */
final class TaxonomyTooltip extends GlossifyBase {

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * {@inheritdoc}
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    LoggerChannelFactoryInterface $logger_factory,
    RendererInterface $renderer,
    CurrentPathStack $currentPath,
    Connection $database,
    EntityFieldManagerInterface $entityFieldManager,
    EntityTypeBundleInfoInterface $entityTypeBundleInfo,
    ModuleHandlerInterface $module_handler,
  ) {
    parent::__construct(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $logger_factory,
      $renderer,
      $currentPath,
      $database,
      $entityFieldManager,
      $entityTypeBundleInfo,
    );
    $this->moduleHandler = $module_handler;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('logger.factory'),
      $container->get('renderer'),
      $container->get('path.current'),
      $container->get('database'),
      $container->get('entity_field.manager'),
      $container->get('entity_type.bundle.info'),
      $container->get('module_handler')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $vocab_options = [];
    $vocabularies = Vocabulary::loadMultiple();
    foreach ($vocabularies as $vocab) {
      $vocab_options[$vocab->id()] = $vocab->get('name');
    }

    $form['glossify_taxonomy_case_sensitivity'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Case sensitive'),
      '#description' => $this->t('Whether or not the match is case sensitive.'),
      '#default_value' => $this->settings['glossify_taxonomy_case_sensitivity'],
    ];
    $form['glossify_taxonomy_first_only'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('First match only'),
      '#description' => $this->t('Match and link only the first occurrence per field.'),
      '#default_value' => $this->settings['glossify_taxonomy_first_only'],
    ];
    $form['glossify_taxonomy_ignore_tags'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Ignore tags'),
      '#description' => $this->t('A comma-separated list of tags to ignore (e.g.: <code>h1,h2,div,strong</code>).'),
      '#default_value' => $this->settings['glossify_taxonomy_ignore_tags'] ?? "",
    ];
    $form['glossify_taxonomy_type'] = [
      '#type' => 'radios',
      '#title' => $this->t('Type'),
      '#required' => TRUE,
      '#options' => [
        'tooltips' => $this->t('Tooltips'),
        'links' => $this->t('Links'),
        'tooltips_links' => $this->t('Tooltips and links'),
      ],
      '#description' => $this->t('How to show matches in content. Description as HTML5 tooltip (abbr element), link to description or both.'),
      '#default_value' => $this->settings['glossify_taxonomy_type'],
    ];
    $form['glossify_taxonomy_tooltip_truncate'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Truncate tooltip'),
      '#description' => $this->t('Whether to truncate tooltip after 300 characters.'),
      '#default_value' => $this->settings['glossify_taxonomy_tooltip_truncate'],
      '#states' => [
        'visible' => [
          ':input[name="filters[glossify_taxonomy][settings][glossify_taxonomy_type]"]' => [
            ['value' => 'tooltips'],
            'or',
            ['value' => 'tooltips_links'],
          ],
        ],
      ],
    ];
    $form['glossify_taxonomy_vocabs'] = [
      '#type' => 'checkboxes',
      '#multiple' => TRUE,
      '#element_validate' => [
        [
          get_class($this),
          'validateTaxonomyVocabs',
        ],
      ],
      '#title' => $this->t('Taxonomy vocabularies'),
      '#description' => $this->t('Select the source taxonomy vocabularies you want to use term names from to link their term page.'),
      '#options' => $vocab_options,
      '#default_value' => explode(';', $this->settings['glossify_taxonomy_vocabs'] ?? ''),
    ];
    $form['glossify_taxonomy_urlpattern'] = [
      '#type' => 'textfield',
      '#title' => $this->t('URL pattern'),
      '#description' => $this->t('Url pattern, used for linking matched words. Accepts "[id]" as token. Example: "/taxonomy/term/[id]"'),
      '#default_value' => $this->settings['glossify_taxonomy_urlpattern'],
    ];
    $form['glossify_taxonomy_synonyms_field'] = [
      '#type' => 'select',
      '#title' => $this->t('Synonyms field'),
      '#description' => $this->t('Select the text field to use for synonyms. Only Text (plain) fields are supported. When a field is selected, not only the taxonomy term name itself will get matched, but also specified synonyms from the selected field. Each field value contains one synonym - configure field cardinality to allow multiple synonyms. <strong>Important:</strong> Make sure the selected field exists on all targeted vocabularies. If the field does not exist on a vocabulary, synonyms will not apply for terms of that vocabulary. Select "None" to disable synonyms matching.'),
      '#options' => $this->getSynonymsFieldOptions('taxonomy_term'),
      '#default_value' => $this->settings['glossify_taxonomy_synonyms_field'] ?? '',
      '#empty_option' => $this->t('None'),
    ];

    return $form;
  }

  /**
   * Validation callback for glossify_taxonomy_vocabs.
   *
   * Make the field required if the filter is enabled.
   *
   * @param array $element
   *   The element being processed.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   * @param array $complete_form
   *   The complete form structure.
   */
  public static function validateTaxonomyVocabs(array &$element, FormStateInterface $form_state, array &$complete_form) {
    $values = $form_state->getValues();
    // Make taxonomy_vocabs required if the filter is enabled.
    if (!empty($values['filters']['glossify_taxonomy']['status'])) {
      $field_values = array_filter($values['filters']['glossify_taxonomy']['settings']['glossify_taxonomy_vocabs']);
      if (empty($field_values)) {
        $element['#required'] = TRUE;
        $form_state->setError($element, t('%field is required.', ['%field' => $element['#title']]));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function process($text, $langcode) {
    $cacheTags = [];

    // Get vocabularies.
    $vocabs = explode(';', $this->settings['glossify_taxonomy_vocabs']);

    // Let other modules override $vocabs.
    $this->moduleHandler->alter('glossify_taxonomy_vocabs', $vocabs);

    if (count($vocabs)) {
      $terms = [];

      // Get taxonomy term data.
      $query = $this->database->select('taxonomy_term_field_data', 'tfd');
      $query->addField('tfd', 'tid', 'id');
      $query->addField('tfd', 'name');
      $query->addField('tfd', 'name', 'name_norm');
      $query->addField('tfd', 'description__value', 'tip');
      $query->condition('tfd.vid', $vocabs, 'IN');
      $query->condition('tfd.status', 1);
      $query->condition('tfd.langcode', $langcode);
      $query->orderBy('name_norm', 'DESC');
      // Let other modules alter the current query.
      $query->addTag('glossify_taxonomy_tooltip');

      $results = $query->execute()->fetchAllAssoc('name_norm');
      // Build terms array.
      foreach ($results as $result) {
        // Make name_norm lowercase, it seems not possible in PDO query?
        if (!$this->settings['glossify_taxonomy_case_sensitivity']) {
          $result->name_norm = mb_strtolower($result->name_norm);
        }
        $terms[$result->name_norm] = $result;
      }

      // Add synonyms if a synonyms field is selected.
      if (!empty($this->settings['glossify_taxonomy_synonyms_field'])) {
        $synonyms = $this->loadSynonyms('taxonomy_term', 'tid', $vocabs, $langcode, 'vid', $this->settings['glossify_taxonomy_synonyms_field']);

        // Group synonyms by entity_id for easier processing.
        $synonyms_by_entity = [];
        foreach ($synonyms as $synonym) {
          // Each field value contains one synonym (respecting field
          // cardinality):
          $synonym_value = trim($synonym->{$this->settings['glossify_taxonomy_synonyms_field'] . '_value'});
          if (!empty($synonym_value)) {
            $synonyms_by_entity[$synonym->entity_id][] = $synonym_value;
          }
        }

        // Add synonyms array to existing terms.
        foreach ($terms as $term) {
          $term->synonyms = $synonyms_by_entity[$term->id] ?? [];
        }
      }

      // Process text.
      if (count($terms) > 0) {
        $text = $this->parseTooltipMatch(
          $text,
          $terms,
          $this->settings['glossify_taxonomy_case_sensitivity'],
          $this->settings['glossify_taxonomy_first_only'],
          $this->settings['glossify_taxonomy_type'],
          $this->settings['glossify_taxonomy_tooltip_truncate'],
          $this->settings['glossify_taxonomy_urlpattern'],
          $this->settings['glossify_taxonomy_ignore_tags'],
          $langcode
        );
      }
    }

    // Prepare result.
    $result = new FilterProcessResult($text);

    // Add cache tag dependency.
    foreach ($vocabs as $vid) {
      $cacheTags[] = 'taxonomy_term_list:' . $vid;
    }
    $result->setCacheTags($cacheTags);
    return $result;
  }

  /**
   * {@inheritdoc}
   */
  public function setConfiguration(array $configuration) {
    if (isset($configuration['status'])) {
      $this->status = (bool) $configuration['status'];
    }
    if (isset($configuration['weight'])) {
      $this->weight = (int) $configuration['weight'];
    }
    if (isset($configuration['settings'])) {
      // Workaround for not accepting arrays in config schema.
      if (is_array($configuration['settings']['glossify_taxonomy_vocabs'])) {
        $glossify_taxonomy_vocabs = array_filter($configuration['settings']['glossify_taxonomy_vocabs']);
        $configuration['settings']['glossify_taxonomy_vocabs'] = implode(';', $glossify_taxonomy_vocabs);
      }
      $this->settings = (array) $configuration['settings'];
    }
    return $this;
  }

}
