(function (Drupal, drupalSettings) {
  const app = document.getElementById('go-app');
  const base = (drupalSettings.goGame?.basePath || '').replace(/\/$/, '');
  let gameId = drupalSettings.goGame?.gameId || null;
  let size = 13, board = [], turn = 1, you = 0, lastMove = null;

  app.innerHTML = `
    <div class="go-wrap">
      <div class="go-topbar">
        <div class="go-brand">Go Multiplayer</div>
        <div class="go-actions">
          <select id="g_size" class="g-select">
            <option value="9">9x9</option>
            <option value="13" selected>13x13</option>
            <option value="19">19x19</option>
          </select>
          <button id="g_new" class="g-btn primary">Create game</button>
          <button id="g_copy" class="g-btn" ${gameId? '' : 'disabled'}>Copy invite</button>
          <button id="g_pass" class="g-btn subtle" disabled>Pass</button>
          <button id="g_resign" class="g-btn danger" disabled>Resign</button>
        </div>
      </div>
      <div class="go-status">
        <div id="g_players" class="go-players"></div>
        <div id="g_turn" class="go-turn">—</div>
        <div id="g_result" class="go-result"></div>
      </div>
      <div class="go-panels">
        <div class="go-panel">
          <div class="go-panel-title">Captures</div>
          <div id="g_captures" class="go-panel-body">B: 0 • W: 0</div>
        </div>
        <div class="go-panel">
          <div class="go-panel-title">Score</div>
          <div id="g_scores" class="go-panel-body">—</div>
        </div>
      </div>
      <div class="go-board-wrap">
        <canvas id="g_board" class="go-canvas"></canvas>
        <div class="go-hint" id="g_hint"></div>
      </div>
      <div id="g_invite" class="go-invite" hidden>
        <div>Share this link to invite a friend:</div>
        <input id="g_link" class="go-link" readonly />
      </div>
      <div id="g_lobby" class="go-lobby" ${gameId? 'hidden' : ''}>
        <div class="go-lobby-title">Recent games</div>
        <div id="g_lobby_list" class="go-lobby-list"></div>
      </div>
    </div>
  `;

  const canvas = document.getElementById('g_board');
  const ctx = canvas.getContext('2d');
  const hint = document.getElementById('g_hint');
  const elTurn = document.getElementById('g_turn');
  const elResult = document.getElementById('g_result');
  const elInvite = document.getElementById('g_invite');
  const elLink = document.getElementById('g_link');
  const dpr = Math.max(1, window.devicePixelRatio || 1);

  function resizeCanvas() {
    const s = Math.min(720, Math.floor(window.innerWidth * 0.9));
    const sizePx = Math.max(320, s);
    canvas.style.width = sizePx + 'px';
    canvas.style.height = sizePx + 'px';
    canvas.width = Math.floor(sizePx * dpr);
    canvas.height = Math.floor(sizePx * dpr);
    draw();
  }
  window.addEventListener('resize', resizeCanvas);

  function setButtonsEnabled(enabled) {
    document.getElementById('g_pass').disabled = !enabled;
    document.getElementById('g_resign').disabled = !enabled;
  }

  function uiStateFrom(data) {
    size = data.size; board = data.board; turn = data.turn; lastMove = data.last_move ?? null; you = data.you || 0;
    const t = (turn === 1 ? 'Black' : 'White');
    const blackName = data.players?.black_name || 'Black';
    const whiteName = data.players?.white_name || 'White';
    document.getElementById('g_players').innerHTML = `
      <span class="p black">${blackName}</span>
      <span class="vs">vs</span>
      <span class="p white">${whiteName}</span>
    `;
    elTurn.textContent = data.status === 'ongoing' ? `${t}'s turn` : 'Game finished';
    if (data.status === 'finished') {
      const sw = data.scores?.white ?? 0; const sb = data.scores?.black ?? 0;
      elResult.textContent = `Winner: ${data.winner === 'draw' ? 'Draw' : data.winner} (B ${sb.toFixed(1)} – W ${sw.toFixed(1)})`;
      setButtonsEnabled(false);
    } else {
      setButtonsEnabled(you === turn && you !== 0);
      elResult.textContent = '';
    }
    const capB = data.captures?.black ?? 0; const capW = data.captures?.white ?? 0;
    document.getElementById('g_captures').textContent = `B: ${capB} • W: ${capW}`;
    const scoreB = data.scores?.black; const scoreW = data.scores?.white; const komi = data.komi ?? 6.5;
    document.getElementById('g_scores').textContent = (scoreB==null || scoreW==null) ? `Komi: ${komi}` : `B: ${scoreB.toFixed(1)} • W: ${scoreW.toFixed(1)} (komi ${komi})`;
    draw();
  }

  function draw() {
    const n = size, w = canvas.width, h = canvas.height;
    const margin = Math.floor(40 * dpr);
    const cell = (w - 2 * margin) / (n - 1);
    ctx.clearRect(0, 0, w, h);

    const grad = ctx.createLinearGradient(0, 0, w, h);
    grad.addColorStop(0, '#e9c57a');
    grad.addColorStop(1, '#d6ae62');
    ctx.fillStyle = grad;
    ctx.fillRect(0, 0, w, h);

    ctx.save(); ctx.translate(0, Math.floor(2 * dpr)); ctx.strokeStyle = 'rgba(0,0,0,0.25)';
    for (let i = 0; i < n; i++) { ctx.beginPath(); ctx.moveTo(margin, margin + i * cell); ctx.lineTo(margin + (n - 1) * cell, margin + i * cell); ctx.stroke(); ctx.beginPath(); ctx.moveTo(margin + i * cell, margin); ctx.lineTo(margin + i * cell, margin + (n - 1) * cell); ctx.stroke(); }
    ctx.restore();

    ctx.strokeStyle = '#1f2937'; ctx.lineWidth = Math.max(1, Math.floor(1 * dpr));
    for (let i = 0; i < n; i++) { ctx.beginPath(); ctx.moveTo(margin, margin + i * cell); ctx.lineTo(margin + (n - 1) * cell, margin + i * cell); ctx.stroke(); ctx.beginPath(); ctx.moveTo(margin + i * cell, margin); ctx.lineTo(margin + i * cell, margin + (n - 1) * cell); ctx.stroke(); }

    const star = []; if (n === 9) star.push(2, 4, 6); if (n === 13) star.push(3, 6, 9); if (n === 19) star.push(3, 9, 15);
    if (star.length) { ctx.fillStyle = '#111827'; star.forEach(r => star.forEach(c => { ctx.beginPath(); ctx.arc(margin + c * cell, margin + r * cell, Math.max(2, Math.floor(2.2 * dpr)), 0, Math.PI * 2); ctx.fill(); })); }

    const radius = cell * 0.46;
    for (let r = 0; r < n; r++) for (let c = 0; c < n; c++) {
      const v = board[r * n + c] || 0; if (!v) continue; const x = margin + c * cell, y = margin + r * cell;
      ctx.save(); ctx.beginPath(); ctx.arc(x, y + 1 * dpr, radius, 0, Math.PI * 2); ctx.fillStyle = 'rgba(0,0,0,0.15)'; ctx.fill(); ctx.restore();
      const grd = ctx.createRadialGradient(x - radius * 0.4, y - radius * 0.4, radius * 0.15, x, y, radius);
      if (v === 1) { grd.addColorStop(0, '#111'); grd.addColorStop(1, '#444'); } else { grd.addColorStop(0, '#fff'); grd.addColorStop(1, '#d1d5db'); }
      ctx.beginPath(); ctx.arc(x, y, radius, 0, Math.PI * 2); ctx.fillStyle = grd; ctx.fill();
    }

    if (lastMove != null) { const r = Math.floor(lastMove / n), c = lastMove % n; const x = margin + c * cell, y = margin + r * cell; ctx.beginPath(); ctx.arc(x, y, radius * 0.25, 0, Math.PI * 2); ctx.fillStyle = '#ef4444'; ctx.fill(); }
  }

  function gameUrl(id) {
    const url = new URL(window.location.href);
    url.pathname = `${base}/go/${id}`;
    url.search = '';
    return url.toString();
  }
  function poll() { if (!gameId) return; fetch(`${base}/go/api/state/${encodeURIComponent(gameId)}`).then(r => r.json()).then(uiStateFrom).catch(()=>{}); }
  function post(url, params) { const body = new URLSearchParams(params || {}).toString(); return fetch(url, { method: 'POST', headers: { 'Content-Type': 'application/x-www-form-urlencoded' }, body }).then(r => r.json()); }
  function tryJoin() { if (!gameId) return; post(`${base}/go/api/join`, { id: gameId }).then(() => poll()); }

  document.getElementById('g_new').addEventListener('click', () => {
    const sel = document.getElementById('g_size');
    const s = parseInt(sel.value, 10) || size;
    post(`${base}/go/api/create`, { size: s }).then(j => {
      size = s; gameId = j.id; you = 1;
      elLink.value = gameUrl(gameId);
      elInvite.hidden = false;
      document.getElementById('g_copy').disabled = false;
      document.getElementById('g_lobby').hidden = true;
  history.replaceState({}, '', `${base}/go/${gameId}`);
      poll();
    });
  });
  document.getElementById('g_copy').addEventListener('click', async () => { if (!gameId) return; try { await navigator.clipboard.writeText(gameUrl(gameId)); hint.textContent = 'Invite link copied!'; } catch(_) {} hint.classList.add('show'); setTimeout(()=> hint.classList.remove('show'), 1200); });
  document.getElementById('g_pass').addEventListener('click', () => { if (!gameId) return; post(`${base}/go/api/pass`, { id: gameId }).then(j => { if (j.finished) elResult.textContent = `Winner: ${j.winner}`; poll(); }); });
  document.getElementById('g_resign').addEventListener('click', () => { if (!gameId) return; post(`${base}/go/api/resign`, { id: gameId }).then(j => { if (j.finished) elResult.textContent = `Winner: ${j.winner}`; poll(); }); });

  function posToCell(x, y) { const rect = canvas.getBoundingClientRect(); const s = canvas.width; const n = size; const margin = Math.floor(40 * dpr); const cell = (s - 2 * margin) / (n - 1); const cx = (x - rect.left) * dpr; const cy = (y - rect.top) * dpr; let c = Math.round((cx - margin) / cell); let r = Math.round((cy - margin) / cell); if (r < 0 || r >= n || c < 0 || c >= n) return null; return { r, c }; }
  canvas.addEventListener('mousemove', (e) => { const rc = posToCell(e.clientX, e.clientY); if (!rc || !gameId) { hint.classList.remove('show'); return; } const idx = rc.r * size + rc.c; if (board[idx]) { hint.classList.remove('show'); return; } hint.textContent = you === 1 ? 'Place black' : (you === 2 ? 'Place white' : 'Spectating'); hint.classList.add('show'); hint.style.left = `${e.clientX + 12}px`; hint.style.top = `${e.clientY + 12}px`; });
  canvas.addEventListener('mouseleave', ()=> hint.classList.remove('show'));
  canvas.addEventListener('click', (e) => { if (!gameId || you === 0 || turn !== you) return; const rc = posToCell(e.clientX, e.clientY); if (!rc) return; const idx = rc.r * size + rc.c; if (board[idx]) return; post(`${base}/go/api/move`, { id: gameId, row: rc.r, col: rc.c }).then(res => { if (res.error) { hint.textContent = res.error.replace(/_/g,' '); hint.classList.add('show'); setTimeout(()=> hint.classList.remove('show'), 900); return; } poll(); }); });

  if (gameId) { elLink.value = gameUrl(gameId); elInvite.hidden = false; tryJoin(); }
  resizeCanvas(); setInterval(poll, 1500); poll();
  // Lobby
  function renderLobby(list) {
    const box = document.getElementById('g_lobby_list');
    if (!box) return;
    if (!list || !list.length) { box.innerHTML = '<div class="go-empty">No recent games</div>'; return; }
    box.innerHTML = list.map(g => {
      const names = `${g.players?.black_name || 'Black'} vs ${g.players?.white_name || 'White'}`;
      const when = new Date(g.updated_at * 1000).toLocaleString();
      const score = (g.scores && (g.scores.black || g.scores.white)) ? ` • B ${g.scores.black.toFixed(1)} – W ${g.scores.white.toFixed(1)}` : '';
  return `<div class="go-lobby-item"><div class="go-lobby-meta">${names} • ${g.size}x${g.size} • ${g.status}${score}</div><div class="go-lobby-right"><span class="go-when">${when}</span><a class="g-btn" href="${base}/go/${g.id}">Continue</a></div></div>`;
    }).join('');
  }
  function loadLobby() { if (gameId) return; fetch(`${base}/go/api/lobby`).then(r=>r.json()).then(data => renderLobby(data.games || [])).catch(()=>{}); }
  if (!gameId) { loadLobby(); setInterval(loadLobby, 10000); }
})(Drupal, drupalSettings);
