<?php

namespace Drupal\go_game\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Database;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

class ApiController extends ControllerBase {
  protected function db() { return Database::getConnection(); }
  protected function sessionId(Request $request): string {
    // Prefer Symfony session; start it if not started.
    $session = $request->getSession();
    if ($session && method_exists($session, 'isStarted') && !$session->isStarted()) {
      $session->start();
    }
    if ($session) {
      $sid = (string) $session->getId();
      if ($sid !== '') { return $sid; }
    }
    // Fallback: derive a stable-ish id from IP + User-Agent.
    $ip = (string) ($request->getClientIp() ?? '');
    $ua = (string) ($request->headers->get('User-Agent') ?? '');
    return substr(hash('sha256', $ip.'|'.$ua), 0, 32);
  }
  protected function newId(): string { return bin2hex(random_bytes(8)); }
  protected function load(string $id): ?array {
    $row = $this->db()->select('go_game', 'g')->fields('g')->condition('id', $id)->execute()->fetchAssoc();
    if(!$row) return NULL; $row['size']=(int)$row['size']; $row['turn']=(int)$row['turn']; $row['passes']=(int)$row['passes'];
    $row['komi'] = isset($row['komi']) ? (float) $row['komi'] : 6.5;
    $row['captures_black'] = isset($row['captures_black']) ? (int) $row['captures_black'] : 0;
    $row['captures_white'] = isset($row['captures_white']) ? (int) $row['captures_white'] : 0;
    $row['score_black'] = isset($row['score_black']) ? (float) $row['score_black'] : 0.0;
    $row['score_white'] = isset($row['score_white']) ? (float) $row['score_white'] : 0.0;
    $row['board']=json_decode($row['board']??'[]', true)?:[]; return $row;
  }
  protected function save(array $game): void {
    $f = $game;
    unset($f['id']); // Do not update primary key.
    $f['board'] = json_encode($game['board']);
    $f['updated_at'] = time();
    $this->db()->update('go_game')->fields($f)->condition('id', $game['id'])->execute();
  }
  protected function createRow(int $size, string $owner): array {
    $id=$this->newId(); $board=array_fill(0,$size*$size,0);
    $account = \Drupal::currentUser();
    $uid = (int) $account->id(); $name = $account->isAuthenticated() ? (string) $account->getAccountName() : 'Guest';
    $now = time();
    $this->db()->insert('go_game')->fields([
      'id'=>$id,'size'=>$size,'board'=>json_encode($board),'turn'=>1,
      'black'=>$owner,'white'=>NULL,
      'black_uid'=>$uid ?: NULL,'white_uid'=>NULL,
      'black_name'=>$name,'white_name'=>NULL,
      'last_move'=>NULL,'created_at'=>$now,'updated_at'=>$now,
      'status'=>'ongoing','winner'=>NULL,'passes'=>0,
      'komi'=>6.5,'captures_black'=>0,'captures_white'=>0,
      'score_black'=>0.0,'score_white'=>0.0,
    ])->execute(); return $this->load($id);
  }

  public function createGame(Request $request): JsonResponse { $size=(int)($request->request->get('size')??13); if(!in_array($size,[9,13,19],true)) $size=13; $sid=$this->sessionId($request); $game=$this->createRow($size,$sid); $base=$request->getBasePath(); return new JsonResponse(['id'=>$game['id'],'size'=>$game['size'],'invite'=>$base.'/go/'.$game['id']]); }
  public function join(Request $request): JsonResponse {
    $id=(string)$request->request->get('id'); if(!preg_match('/^[0-9a-f]{16}$/',$id)) return new JsonResponse(['error'=>'invalid_id'],400); $sid=$this->sessionId($request); $game=$this->load($id);
    if(!$game) return new JsonResponse(['error'=>'not_found'],404);
    if($game['status']!=='ongoing') return new JsonResponse(['error'=>'finished'],400);
    $account = \Drupal::currentUser(); $uid = (int) $account->id(); $name = $account->isAuthenticated() ? (string) $account->getAccountName() : 'Guest';
    if($game['black']!==$sid && $game['white']!==$sid){
      if(empty($game['white'])){ $game['white']=$sid; $game['white_uid']=$uid ?: NULL; $game['white_name']=$name; $this->save($game); }
      elseif(empty($game['black'])){ $game['black']=$sid; $game['black_uid']=$uid ?: NULL; $game['black_name']=$name; $this->save($game); }
      else { return new JsonResponse(['error'=>'full'],409); }
    }
    return new JsonResponse([
      'ok'=>true,
      'you'=>($game['black']===$sid)?1:(($game['white']===$sid)?2:0),
      'size'=>$game['size']
    ]);
  }
  public function gameState(string $id, Request $request): JsonResponse {
    if(!preg_match('/^[0-9a-f]{16}$/',$id)) return new JsonResponse(['error'=>'invalid_id'],400);
    $sid=$this->sessionId($request); $g=$this->load($id); if(!$g) return new JsonResponse(['error'=>'not_found'],404);
    return new JsonResponse([
      'id'=>$g['id'],'size'=>$g['size'],'board'=>$g['board'],'turn'=>$g['turn'],
      'status'=>$g['status'],'winner'=>$g['winner'],'last_move'=>$g['last_move'],
      'you'=>($g['black']===$sid)?1:(($g['white']===$sid)?2:0),
      'players'=>[
        'black'=>!empty($g['black']), 'white'=>!empty($g['white']),
        'black_name'=>$g['black_name'] ?? 'Black', 'white_name'=>$g['white_name'] ?? 'White',
      ],
      'komi'=>$g['komi'],
      'captures'=>['black'=>$g['captures_black'],'white'=>$g['captures_white']],
      'scores'=>['black'=>$g['score_black'],'white'=>$g['score_white']],
    ]);
  }
  public function move(Request $request): JsonResponse {
    $id=(string)$request->request->get('id'); if(!preg_match('/^[0-9a-f]{16}$/',$id)) return new JsonResponse(['error'=>'invalid_id'],400); $row=(int)$request->request->get('row'); $col=(int)$request->request->get('col');
    $sid=$this->sessionId($request); $g=$this->load($id);
    if(!$g) return new JsonResponse(['error'=>'not_found'],404);
    if($g['status']!=='ongoing') return new JsonResponse(['error'=>'finished'],400);
    $you=($g['black']===$sid)?1:(($g['white']===$sid)?2:0);
    if($you===0) return new JsonResponse(['error'=>'not_joined'],403);
    if($you!==$g['turn']) return new JsonResponse(['error'=>'not_your_turn'],400);
    $n=$g['size']; if($row<0||$row>=$n||$col<0||$col>=$n) return new JsonResponse(['error'=>'out_of_bounds'],400);
    $idx=$row*$n+$col; if(!empty($g['board'][$idx])) return new JsonResponse(['error'=>'occupied'],400);
    $b=$g['board']; $b[$idx]=$you; $capt=$this->applyCaptures($b,$n,$row,$col,3-$you);
    if(!$this->hasLiberty($b,$n,$row,$col)) return new JsonResponse(['error'=>'suicide'],400);
    $g['board']=$b; $g['turn']=3-$g['turn']; $g['passes']=0; $g['last_move']=$idx;
    if ($capt>0) {
      if ($you===1) $g['captures_black'] += $capt; else $g['captures_white'] += $capt;
    }
    $this->save($g);
    return new JsonResponse(['ok'=>true,'captured'=>$capt]);
  }
  public function pass(Request $request): JsonResponse {
    $id=(string)$request->request->get('id'); if(!preg_match('/^[0-9a-f]{16}$/',$id)) return new JsonResponse(['error'=>'invalid_id'],400); $sid=$this->sessionId($request); $g=$this->load($id);
    if(!$g) return new JsonResponse(['error'=>'not_found'],404);
    $you=($g['black']===$sid)?1:(($g['white']===$sid)?2:0);
    if($you!==$g['turn']) return new JsonResponse(['error'=>'not_your_turn'],400);
    if($g['status']!=='ongoing') return new JsonResponse(['error'=>'finished'],400);
    $g['passes']++; $g['turn']=3-$g['turn']; $g['last_move']=NULL;
    if($g['passes']>=2){
      // Compute territory + captures + komi scoring.
      [$sb, $sw] = $this->scoreGame($g['board'], $g['size'], $g['captures_black'], $g['captures_white'], $g['komi']);
      $g['score_black'] = $sb; $g['score_white'] = $sw;
      $g['status']='finished';
      $g['winner'] = ($sb > $sw) ? 'black' : (($sw > $sb) ? 'white' : 'draw');
    }
    $this->save($g);
    return new JsonResponse(['ok'=>true,'finished'=>$g['status']==='finished','winner'=>$g['winner'],'scores'=>['black'=>$g['score_black'],'white'=>$g['score_white']]]);
  }
  public function resign(Request $request): JsonResponse { $id=(string)$request->request->get('id'); if(!preg_match('/^[0-9a-f]{16}$/',$id)) return new JsonResponse(['error'=>'invalid_id'],400); $sid=$this->sessionId($request); $g=$this->load($id); if(!$g) return new JsonResponse(['error'=>'not_found'],404); $you=($g['black']===$sid)?1:(($g['white']===$sid)?2:0); if($you===0) return new JsonResponse(['error'=>'not_joined'],403); if($g['status']!=='ongoing') return new JsonResponse(['error'=>'finished'],400); $g['status']='finished'; $g['winner']=($you===1)?'white':'black'; $this->save($g); return new JsonResponse(['ok'=>true,'finished'=>true,'winner'=>$g['winner']]); }

  public function lobby(Request $request): JsonResponse {
    $q = $this->db()->select('go_game','g')->fields('g',['id','size','status','winner','created_at','updated_at','black_name','white_name','score_black','score_white'])->orderBy('updated_at','DESC')->range(0,20);
    $rows = $q->execute()->fetchAllAssoc('id');
    $list = [];
    foreach ($rows as $r) {
      $list[] = [
        'id'=>$r->id,
        'size'=>(int)$r->size,
        'status'=>$r->status,
        'winner'=>$r->winner,
        'created_at'=>(int)$r->created_at,
        'updated_at'=>(int)$r->updated_at,
        'players'=>['black_name'=>$r->black_name,'white_name'=>$r->white_name],
        'scores'=>['black'=>(float)$r->score_black,'white'=>(float)$r->score_white],
      ];
    }
    return new JsonResponse(['games'=>$list]);
  }

  private function neighbors(int $n, int $r, int $c): array { $o=[]; if($r>0)$o[]=[$r-1,$c]; if($r<$n-1)$o[]=[$r+1,$c]; if($c>0)$o[]=[$r,$c-1]; if($c<$n-1)$o[]=[$r,$c+1]; return $o; }
  private function hasLiberty(array $b,int $n,int $r,int $c): bool { $color=$b[$r*$n+$c]??0; if(!$color) return false; $vis=[]; $st=[[$r,$c]]; while($st){ [$rr,$cc]=array_pop($st); $k=$rr.'-'.$cc; if(isset($vis[$k])) continue; $vis[$k]=true; foreach($this->neighbors($n,$rr,$cc) as [$nr,$nc]){ $idx=$nr*$n+$nc; $v=$b[$idx]??0; if($v===0) return true; if($v===$color) $st[]=[$nr,$nc]; } } return false; }
  private function collectGroup(array $b,int $n,int $r,int $c): array { $color=$b[$r*$n+$c]??0; $grp=[]; $st=[[$r,$c]]; while($st){ [$rr,$cc]=array_pop($st); $k=$rr.'-'.$cc; if(isset($grp[$k])) continue; if(($b[$rr*$n+$cc]??0)!==$color) continue; $grp[$k]=[$rr,$cc]; foreach($this->neighbors($n,$rr,$cc) as [$nr,$nc]) if(($b[$nr*$n+$nc]??0)===$color) $st[]=[$nr,$nc]; } return array_values($grp); }
  private function removeIfNoLiberty(array &$b,int $n,int $r,int $c): int { if(($b[$r*$n+$c]??0)===0) return 0; if($this->hasLiberty($b,$n,$r,$c)) return 0; $grp=$this->collectGroup($b,$n,$r,$c); $rm=0; foreach($grp as [$gr,$gc]){ $b[$gr*$n+$gc]=0; $rm++; } return $rm; }
  private function applyCaptures(array &$b,int $n,int $r,int $c,int $enemy): int { $rm=0; foreach($this->neighbors($n,$r,$c) as [$nr,$nc]) if(($b[$nr*$n+$nc]??0)===$enemy) $rm+=$this->removeIfNoLiberty($b,$n,$nr,$nc); return $rm; }

  // --- Scoring (territory + captures + komi) ---
  private function scoreGame(array $board, int $n, int $capB, int $capW, float $komi): array {
    $visited = array_fill(0, $n*$n, false);
    $territoryB = 0; $territoryW = 0;
    for ($r=0; $r<$n; $r++) {
      for ($c=0; $c<$n; $c++) {
        $idx = $r*$n+$c; $v = $board[$idx] ?? 0; if ($v !== 0 || $visited[$idx]) continue;
        // BFS empty region
        $queue = [[$r,$c]]; $region = [[$r,$c]]; $visited[$idx] = true; $border = [];
        while ($queue) {
          [$rr,$cc] = array_shift($queue);
          foreach ($this->neighbors($n,$rr,$cc) as [$nr,$nc]) {
            $ii = $nr*$n+$nc; $vv = $board[$ii] ?? 0;
            if ($vv === 0) { if (!$visited[$ii]) { $visited[$ii]=true; $queue[] = [$nr,$nc]; $region[] = [$nr,$nc]; } }
            else { $border[$vv] = true; }
          }
        }
        if (isset($border[1]) && !isset($border[2])) $territoryB += count($region);
        elseif (isset($border[2]) && !isset($border[1])) $territoryW += count($region);
        // mixed borders => neutral
      }
    }
    $scoreB = $territoryB + $capB; $scoreW = $territoryW + $capW + $komi;
    return [$scoreB, $scoreW];
  }
}
