<?php

namespace Drupal\google_pagemap\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\field\FieldStorageConfigInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Settings form for Google PageMaps.
 */
class GooglePageMapSettingsForm extends ConfigFormBase {

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected EntityFieldManagerInterface $entityFieldManager;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public function __construct(ConfigFactoryInterface $config_factory, EntityTypeManagerInterface $entity_type_manager, EntityFieldManagerInterface $entity_field_manager) {
    parent::__construct($config_factory);
    $this->entityTypeManager = $entity_type_manager;
    $this->entityFieldManager = $entity_field_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('entity_type.manager'),
      $container->get('entity_field.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['google_pagemap.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'google_pagemap_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildForm($form, $form_state);
    $config = $this->config('google_pagemap.settings');
    $nodeTypes = $this->entityTypeManager->getStorage('node_type')->loadMultiple();

    $options = [];
    foreach ($nodeTypes as $nodeType) {
      $options[$nodeType->id()] = $nodeType->label();
    }
    $form['node_types'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Node types'),
      '#options' => $options,
      '#default_value' => $config->get('node_types') ?? [],
      '#description' => $this->t('Select any types that should have a
        PageMap included. If there are no supported fields associated with that
        node type, then the only attribute will be the Content-Type.'),
    ];

    foreach ($nodeTypes as $nodeType) {
      $nodeTypeFieldOptions = $this->getNodeTypeFieldOptions($nodeType);
      if (!empty($nodeTypeFieldOptions)) {
        $form[$nodeType->id()] = [
          '#title' => $nodeType->label(),
          '#type' => 'details',
          '#open' => FALSE,
          '#states' => [
            'invisible' => [
              ':input[name="node_types[' . $nodeType->id() . ']"]' => ['checked' => FALSE],
            ],
          ],
        ];

        $form[$nodeType->id()][$nodeType->id() . '_fields'] = [
          '#type' => 'checkboxes',
          '#title' => $this->t('Fields To Map'),
          '#options' => $nodeTypeFieldOptions,
          '#default_value' => $config->get($nodeType->id() . '.fields') ?? [],
        ];
      }
    }

    $form['actions'] = ['#type' => 'actions'];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $config = $this->config('google_pagemap.settings');
    $config->set('node_types', $form_state->getValue('node_types'));
    foreach ($form_state->getValue('node_types') as $node_type) {
      if ($node_type) {
        $config->set($node_type . '.fields', $form_state->getValue($node_type . '_fields'));
      }
    }
    $config->save();

  }

  /**
   * Returns fields that can be used in a PageMap for a node type.
   *
   * @param \Drupal\Core\Entity\EntityInterface $nodeType
   *   The node type.
   *
   * @return array
   *   An array of fields to be used as options.
   */
  protected function getNodeTypeFieldOptions(EntityInterface $nodeType): array {
    $nodeTypeFieldOptions = [];
    $fields = $this->entityFieldManager->getFieldDefinitions('node', $nodeType->id());
    foreach ($fields as $field) {
      $allowedTypes = [
        'entity_reference',
        'language',
        'list_string',
        'string',
        'string_long',
        'boolean',
        'text_long',
      ];

      // Check for fields that are being used.
      $storedFieldNames = [];
      $storedFields = $this->entityTypeManager->getStorage('field_storage_config')->loadMultiple();
      foreach ($storedFields as $storedField) {
        if ($storedField instanceof FieldStorageConfigInterface) {
          $storedFieldNames[] = $storedField->getName();
        }
      }

      // Field is allowed and in use.
      if (in_array($field->getType(), $allowedTypes) && in_array($field->getName(), $storedFieldNames)) {
        $nodeTypeFieldOptions[$field->getName()] = $field->getName();
      }
    }

    return $nodeTypeFieldOptions;
  }

}
