<?php

namespace Drupal\google_pagemap\Plugin\preprocessors;

use Drupal\Core\Field\EntityReferenceFieldItemListInterface;
use Drupal\node\NodeInterface;
use Drupal\preprocessors\PreprocessorPluginBase;
use Drupal\text\Plugin\Field\FieldType\TextFieldItemList;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provide plugin to preprocess variable for node.
 *
 * @Preprocessor (
 *   id = "google_pagemap.preprocessor.html",
 *   hooks = {
 *     "html",
 *   },
 *   themes = "*",
 *   weight = 0,
 * )
 */
final class HtmlPreprocessor extends PreprocessorPluginBase {

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\Renderer
   */
  protected $renderer;

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The route match service.
   *
   * @var \Drupal\Core\Routing\CurrentRouteMatch
   */
  protected $routeMatch;

  /**
   * {@inheritdoc}
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, $renderer, $config_factory, $route_match) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->renderer = $renderer;
    $this->configFactory = $config_factory;
    $this->routeMatch = $route_match;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('renderer'),
      $container->get('config.factory'),
      $container->get('current_route_match')
    );
  }

  /**
   * Preprocess the variables.
   *
   * This method works just like a 'hook_preprocess_HOOK()' function.
   *
   * {@inheritdoc}
   */
  public function preprocess(array &$variables, string $hook, array $info): void {

    $pagemap = $this->getPageMap();
    if (empty($pagemap)) {
      return;
    }

    $renderedPageMap = $this->renderer->render($pagemap);

    $variables['#attached']['html_head'][] = [
      [
        '#type' => 'markup',
        '#markup' => $renderedPageMap,
      ],
      'google_pagemap',
    ];
  }

  /**
   * Builds and returns the PageMap.
   *
   * @return array|null
   *   The formatted PageMap.
   */
  private function getPageMap(): ?array {
    $attributes = [];
    $config = $this->configFactory->getEditable('google_pagemap.settings');
    if (($node = $this->routeMatch->getParameter('node')) && $node instanceof NodeInterface) {
      $nodeTypes = $config->get('node_types');
      if (!$nodeTypes || !$nodeTypes[$node->bundle()]) {
        return NULL;
      }
      $fields = $config->get($node->bundle() . '.fields');
      $attributes = $this->getPageMapAttributes($node, $fields);
    }
    if (empty($attributes)) {
      return NULL;
    }

    return [
      '#theme' => 'pagemap',
      '#attributes' => $attributes,
    ];
  }

  /**
   * Builds and returns the attributes for the PageMap.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node for this page.
   * @param array|null $fields
   *   The fields for this node.
   *
   * @return array
   *   The Attribute tags, line-by-line.
   */
  private function getPageMapAttributes(NodeInterface $node, ?array $fields): array {
    $attributes = [];
    // First add the content type as an attribute.
    $attributes[] = [
      'name' => 'content_type',
      'value' => $node->bundle(),
    ];
    if (empty($fields)) {
      return $attributes;
    }
    // Add the selected fields as attributes.
    foreach ($fields as $field) {
      if ($field) {
        $name = $field;
        $values = [];

        // Check if field exists (in case it was removed).
        if ($node->hasField($name) === FALSE) {
          continue;
        };

        // Handle entity reference fields.
        if ($node->get($name) instanceof EntityReferenceFieldItemListInterface) {
          $refField = $node->get($name);
          /** @var \Drupal\Core\Field\EntityReferenceFieldItemListInterface $refField */
          $entities = $refField->referencedEntities();
          if (!empty($entities)) {
            foreach ($entities as $entity) {
              $values[] = $entity->label();
            }
          }
        }
        else if($node->get($name) instanceof TextFieldItemList) {
            $textLongField = $node->get($name);
            /** @var \Drupal\text\Plugin\Field\FieldType\TextFieldItemList $textLongField */

            $textValues = $textLongField->getValue();
            foreach ($textValues as $textValue) {
                $values[] = strip_tags($textValue['value']);
            }
        }
        else {
          $fieldValues = $node->get($name)->getValue();
          foreach ($fieldValues as $fieldValue) {
            $values[] = $fieldValue['value'];
          }
        }
        if (!empty($values)) {

          foreach ($values as $value) {
            $attributes[] = [
              'name' => preg_replace('/field_/', '', $name, 1),
              'value' => $value,
            ];
          }

        }
      }
    }
    return $attributes;
  }

}
