<?php

namespace Drupal\gotenberg\Plugin\EntityPrint\PrintEngine;

use Drupal\entity_print\Plugin\EntityPrint\PrintEngine\PdfEngineBase;
use Drupal\gotenberg\Gotenberg as GotenbergWrapper;
use Gotenberg\Stream;
use Psr\Http\Message\RequestInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Response;

/**
 * A Entity Print plugin for the DomPdf library.
 *
 * @PrintEngine(
 *   id = "gotenberg",
 *   label = @Translation("Gotenberg"),
 *   export_type = "pdf"
 * )
 */
class Gotenberg extends PdfEngineBase {

  protected GotenbergWrapper $gotenberg;

  protected string $html = '';

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->gotenberg = $container->get('gotenberg');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function addPage($content): static {
    // We must keep adding to previously added HTML as loadHtml() replaces the
    // entire document.
    $this->html .= (string) $content;

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function send($filename, $force_download = TRUE) {
    $pdf_request = $this->getPdfRequest();
    $pdf_response = $this->gotenberg->send($pdf_request);

    $html_response = new Response(
      $pdf_response->getBody()->getContents(),
      200,
      [
        'Content-Type' => 'application/pdf',
        'Content-disposition' => ($force_download ? 'attachment; ' : '') . 'filename="' . $filename . '"',
      ],
    );

    $html_response->send();
  }

  /**
   * {@inheritdoc}
   */
  public function getBlob() {
    $pdf_request = $this->getPdfRequest();
    $response = $this->gotenberg->send($pdf_request);
    return $response->getBody()->getContents();
  }

  /**
   * {@inheritdoc}
   */
  protected function getPaperSizes() {
    return array_combine(array_keys(GotenbergWrapper::PAPER_SIZES), array_map('ucfirst', array_keys(GotenbergWrapper::PAPER_SIZES)));
  }

  /**
   * {@inheritDoc}
   */
  public static function dependenciesAvailable() {
    return class_exists('Gotenberg\Gotenberg') && !drupal_valid_test_ua();
  }

  /**
   * {@inheritDoc}
   */
  public function getPrintObject() {
    $print_object = $this->gotenberg->chromium()
      ->pdf();

    if (isset(GotenbergWrapper::PAPER_SIZES[$this->configuration['default_paper_size']])) {
      [$width, $height] = GotenbergWrapper::PAPER_SIZES[$this->configuration['default_paper_size']];
      $print_object = $print_object->paperSize($width, $height);
    }

    if ($this->configuration['default_paper_orientation'] === 'landscape') {
      $print_object = $print_object->landscape();
    }

    return $print_object;
  }

  /**
   * Retrieves the PDF request.
   */
  private function getPdfRequest(): RequestInterface {
    return $this->getPrintObject()
      ->html(Stream::string('pdf.html', $this->html));
  }

}
