<?php

declare(strict_types=1);

namespace Drupal\graphql\EventSubscriber;

use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\Translator\TranslatorInterface;
use Drupal\language\ConfigurableLanguageManagerInterface;
use Drupal\language\LanguageNegotiatorInterface;

/**
 * Sets the current language for the current request.
 */
trait CurrentLanguageResetTrait {

  /**
   * The language manager.
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * The language negotiator.
   */
  protected ?LanguageNegotiatorInterface $languageNegotiator = NULL;

  /**
   * The translator.
   */
  protected TranslatorInterface $translator;

  /**
   * The current user service.
   */
  protected AccountInterface $currentUser;

  /**
   * Resets the global language context across different services.
   */
  protected function resetLanguageContext(): void {
    if (!isset($this->languageNegotiator)) {
      return;
    }

    if (!$this->languageManager->isMultilingual()) {
      return;
    }

    $this->languageNegotiator->setCurrentUser($this->currentUser);
    if ($this->languageManager instanceof ConfigurableLanguageManagerInterface) {
      $this->languageManager->setNegotiator($this->languageNegotiator);
      $this->languageManager->setConfigOverrideLanguage($this->languageManager->getCurrentLanguage());
    }

    // After the language manager has initialized, set the default langcode for
    // the string translations.
    if (method_exists($this->translator, 'setDefaultLangcode')) {
      $language = $this->languageManager->getCurrentLanguage()->getId();
      $this->translator->setDefaultLangcode($language);
    }
  }

}
