# Graphql Commerce

This module extends the [GraphQL core schema module](https://www.drupal.org/project/graphql_core_schema) to support the [Commerce Core module](https://www.drupal.org/project/commerce).

## Dev Notes

The module is in the development phase with shopping cart functionalities already implemented, while the checkout process is pending development and integration. Stability and full functionality are goals yet to be achieved as development progresses.

| Schema Extension | Description |
| --- | --- |
| Commerce | Implements most cart functions. |
| Checkout | Implements Checkout functions. (under development) |
| Commerce Attributes | Provides the attribute fields. |
| Commerce Availability | Gives access to the Availability Checker Resolver. |
| Price | Provides raw and formatted prices. |
| Promotion | Implements the Promotion functionality. |
| Registration | Implements [Registration](https://www.drupal.org/project/commerce_registration) functionality. |

## Endpoints

### commerceGetCart
Gets a cart for a given account, store and order type. 

##### Arguments
  1. `create`: Only used when there's no existing cart found. If `true` it creates a new cart otherwise it returns a `NULL` order.
  1. `orderType`: The commerce order bundle string, like `"default"`.
  1. `account`: The drupal user id. If `""` then the current user is used.
  1. `store`: The commerce store id. If `""` then the current store is used. Note that some modules (e.g. commerce_stock) tends to use the current store  even if this argument is set. Use the `Graphql-Commerce-Current-Store` request header to set the current store uuid for such cases.

##### Response
A `CommerceOrderResponse` object with an `errors` and an `order` property. See the examples below.

##### Example query
    query ExampleCommerceGetCart($create: Boolean!, $orderType: String!, $account: ID!, $store: ID!) {
      commerceGetCart(create: $create, orderType: $orderType, account: $account, store: $store) {
        errors
        order {
          id
        }
      }
    }

Variables:

    {
      "create": true,
      "orderType": "default",
      "account": "112",
      "store": ""
    }
    
Response: 

    {
      "data": {
        "commerceGetCart": {
          "errors": [],
          "order": {
            "id": "8"
          }
        }
      }
    }

Trying to get cart for a nonexistent order type:

    {
      "create": true,
      "orderType": "no_such_bundle",
      "account": "112",
      "store": ""
    }

Response: 

    {
      "data": {
        "commerceGetCart": {
          "errors": [
            {
              "message": "This order type does not exist."
            }
          ],
          "order": null
        }
      }
    }

### commerceGetCarts
Gets carts optionally filtered by account, store and order type. 

##### Arguments
  1. `orderType`: The commerce order bundle string, like `"default"`.
  1. `accountId`: The drupal user id to filter by.
  1. `storeId`: The commerce store id to filter by.

##### Response
An array of `CommerceOrder` objects.

##### Example query
    query ExampleCommerceGetCarts($orderType: String, $accountId: ID, $storeId: ID) {
      commerceGetCart(orderType: $orderType, accountId: $accountId, storeId: $storeId) {
        id
      }
    }

Variables:

    {
      "orderType": "default",
      "account": "112",
      "store": ""
    }
    
Response: 

    {
      "data": {
        "commerceGetCarts":  [
          {
            "id": "5"
          },
          {
            "id": "8"
          }
        ]
      }
    }

### commerceAddToCart

Adds an item to a cart.

##### Arguments
  1. `order`: The cart commerce_order entity id to add the item to. Setting it to `""` will get the cart based on `orderType`, `store` and `account`.
  1. `productVariation`: The variation to add to the cart.
  1. `quantity`: The quantity to add to the cart.
  1. `orderType`: The commerce order bundle string, like `"default"`. Used only if `order` is `""`.
  1. `account`: The drupal user id. Used only if `order` is `""`. If `""` then the current user is used.
  1. `store`: The commerce store id. Used only if `order` is `""`. If `""` then the current store is used. Note that some modules (e.g. commerce_stock) tends to use the current store  even if this argument is set. Use the `Graphql-Commerce-Current-Store` request header to set the current store uuid for such cases.

##### Response
A `CommerceOrderResponse` object with an `errors` and an `order` property. See the examples below.

##### Example query
```
mutation exampleCommerceAddToCart($order: ID, $account: ID, $orderType: String, $productVariation: ID!, $quantity: Int!, $store: ID) {
  commerceAddToCart(
    quantity: $quantity
    productVariation: $productVariation
    store: $store
    orderType: $orderType
    order: $order
    account: $account
  ) {
    errors
    order {
      id
      orderItems {
        quantity
        purchasedEntity {
          id
          title
        }
      }
    }
  }
}

```
Variables: 

```
{
  "quantity": 1,
  "productVariation": 1,
  "store": "",
  "orderType": "default",
  "order": "",
  "account": 1
```

Response:

```
{
  "data": {
    "commerceAddToCart": {
      "errors": [],
      "order": {
        "id": "8",
        "orderItems": [
          {
            "quantity": 21,
            "purchasedEntity": {
              "id": "2",
              "title": "P1"
            }
          },
          {
            "quantity": 25,
            "purchasedEntity": {
              "id": "1",
              "title": "P1"
            }
          }
        ]
      }
    }
  }
}
```

### commerceRemoveFromCart

Removes an item from an order. There's no validation made if the order is a cart or not. 

Can also be used for orphan items (i.e. if the item does not belong to any order). In this case it only removes the item.

##### Arguments
  1. `orderItem`: The order item id to remove.

##### Response
A `CommerceOrderResponse` object with an `errors` and an `order` property. See the examples below.

##### Example query
    mutation exampleCommerceRemoveFromCart {
      commerceRemoveFromCart(orderItem: "4") {
        errors
        order {
          id
        }
      }
    }

Response: 

    {
      "data": {
        "commerceRemoveFromCart": {
          "errors": [],
          "order": {
            "id": "8"
          }
        }
      }
    }

### commerceEmptyCart

Removes all order items from an order. There's no validation made if the order is a cart or not. 

##### Arguments
  1. `order`: The order to empty.

##### Response
A `CommerceOrderResponse` object with an `errors` and an `order` property. See the examples below.

##### Example query
    mutation exampleCommerceEmptyCart {
      commerceEmptyCart(order: "4") {
        errors
        order {
          id
          orderItems {
            id
          }
        }
      }
    }

Response: 

    {
      "data": {
        "commerceEmptyCart": {
          "errors": [],
          "order": {
            "id": "4"
            "orderItems": []
          }
        }
      }
    }

### commerceCartChangeQuantity

Changes an order item quantity.

##### Arguments
  1. `item`: The order item id.
  1. `quantity`: The quantity to change to.

##### Response
A `CommerceOrderResponse` object with an `errors` and an `order` property. See the examples below.

##### Example query
    mutation exampleCommerceCartChangeQuantity {
      commerceCartChangeQuantity(quantity: 29, item: "7") {
        errors
        order {
          orderItems {
            purchasedEntity {
              label
            }
            label
            quantity
          }
        }
      }
    }
    
Response:

    {
      "data": {
        "commerceCartChangeQuantity": {
          "errors": [],
          "order": {
            "orderItems": [
              {
                "purchasedEntity": {
                  "label": "Product 1"
                },
                "label": "Product 1",
                "quantity": 29
              },
              {
                "purchasedEntity": {
                  "label": "Product 2"
                },
                "label": "Product 2",
                "quantity": 11
              }
            ]
          }
        }
      }
    }
