<?php

namespace Drupal\graphql_commerce_promotion\Plugin\GraphQL\DataProducer;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\Core\Render\RenderContext;
use Drupal\graphql\GraphQL\Execution\FieldContext;
use Drupal\graphql_commerce\Plugin\GraphQL\CommerceCartDataProducerBase;
use Drupal\graphql_commerce\Wrapper\CommerceOrderResponse;

/**
 * Remove a coupon from the given order.
 *
 * @DataProducer(
 *   id = "commerce_promotion_remove_coupon",
 *   name = @Translation("Commerce Promotion: Remove Coupon"),
 *   description = @Translation("Removes a coupon on the given order."),
 *   produces = @ContextDefinition("any",
 *     label = @Translation("The mutation result.")
 *   ),
 *   consumes = {
 *     "couponCode" = @ContextDefinition("string",
 *       label = @Translation("The coupon to remove."),
 *     ),
 *     "order" = @ContextDefinition("entity:commerce_order",
 *       label = @Translation("The commerce order."),
 *       required = FALSE,
 *     )
 *   }
 * )
 */
class RemoveCoupon extends CommerceCartDataProducerBase {

  /**
   * The resolver.
   *
   * @param string $couponCode
   *   The coupon to remove.
   * @param \Drupal\commerce_order\Entity\Order $order
   *   The order.
   * @param \Drupal\graphql\GraphQL\Execution\FieldContext $field
   *   The field context.
   *
   * @return \Drupal\graphql_commerce\Wrapper\CommerceOrderResponse
   *   The result.
   */
  public function resolve(string $couponCode, ?OrderInterface $order, ?FieldContext $field): CommerceOrderResponse {
    $context = new RenderContext();

    $result = $this->renderer->executeInRenderContext(
      $context, function () use ($order, $couponCode) {
        if (!$order) {
          $order = $this->cartProvider->getCart('default');
          if (!$order) {
            $order = $this->cartProvider->createCart('default');
          }
        }
        $result = new CommerceOrderResponse();
        $result->setOrder($order);

        /** @var \Drupal\commerce_promotion\CouponStorageInterface $couponStorage */
        $couponStorage = $this->entityTypeManager->getStorage('commerce_promotion_coupon');
        $coupon = $couponStorage->loadEnabledByCode($couponCode);

        // Check if coupon exists.
        if (empty($coupon)) {
          $result->addViolation($this->t('The provided coupon code is invalid.'));
          return $result;
        }
        $id = $coupon->id();
        $couponIds = array_column($order->get('coupons')->getValue(), 'target_id');
        $index = array_search($id, $couponIds);

        if ($index === FALSE) {
          $result->addViolation($this->t('The provided coupon has not been applied to this order.'));
          return $result;
        }

        $order->get('coupons')->removeItem($index);
        $order->save();

        return $result;
      }
    );

    $field->addCacheableDependency($context);
    return $result;
  }

}
