<?php

declare(strict_types=1);

namespace Drupal\Tests\graphql_commerce_promotion\Kernel\SchemaExtension;

use Drupal\graphql\Entity\ServerInterface;
use Drupal\Tests\graphql_commerce_promotion\Kernel\CouponTestBase;
use GraphQL\Server\OperationParams;

/**
 * Tests the commerce_promotion extension.
 *
 * @coversDefaultClass \Drupal\graphql_commerce_promotion\Plugin\GraphQL\SchemaExtension\PromotionExtension
 * @group graphql_core_schema
 */
class PromotionExtensionTest extends CouponTestBase {

  /**
   * The GraphQL server.
   */
  protected ServerInterface $server;

  /**
   * Test the coupon resolvers.
   */
  public function testCouponExtensions(): void {
    $cart = $this->cartProvider->createCart('default');
    $this->cartManager->addEntity($cart, $this->variation, 1);

    $this->server = $this
      ->getCoreComposableServerBuilder()
      ->enableValueFields()
      ->enableExtension('commerce')
      ->enableExtension('commerce_promotion')
      ->enableEntityType('commerce_order', ['id', 'total_price'], ['default'])
      ->enableEntityType('commerce_product_variation', [], ['default'])
      ->enableEntityType('commerce_promotion', ['promotion_id'], ['commerce_promotion'])
      ->enableEntityType('commerce_promotion_coupon', ['code', 'status'], ['commerce_promotion_coupon'])
      ->enableEntityType('user', ['uid'])
      ->createServer();

    // Try a nonexistent coupon.
    $variables = [
      'order' => $cart->id(),
      'couponCode' => 'aaa',
    ];
    $expected = [
      'order' => [
        'id' => 1,
        'totalPrice' => [
          'number' => $this->variation->getPrice()->getNumber(),
        ],
      ],
      'errors' => [
        ['message' => 'The provided coupon code is invalid.'],
      ],
    ];
    $this->assertApplyCoupon($variables, $expected);

    // Add something to the cart for real.
    $variables['couponCode'] = $this->coupon->getCode();
    $expected['errors'] = [];
    $expected['order'] = [
      'id' => 1,
      'totalPrice' => [
        'number' => $this->variation->getPrice()->getNumber() - 25,
      ],
    ];
    $this->assertApplyCoupon($variables, $expected);

    $expected['order'] = [
      'id' => 1,
      'totalPrice' => [
        'number' => $this->variation->getPrice()->getNumber(),
      ],
    ];
    $this->assertRemoveCoupon($variables, $expected);
  }

  /**
   * Assert the applied coupon.
   *
   * @param array $variables
   *   The variables to pass to the query.
   * @param array $expected
   *   The expected values.
   */
  private function assertApplyCoupon(array $variables, array $expected) {
    $query = <<<GQL
    mutation applyCoupon(\$order: ID, \$couponCode: String!) {
      commerceApplyCoupon(
        couponCode: \$couponCode
        order: \$order
      ) {
         order {
          id
          totalPrice {
            number
          }
        }
        errors
      }
    }
    GQL;

    $params = OperationParams::create(
      [
        'query' => $query,
        'variables' => $variables,
      ]
    );
    $result = $this->server->executeOperation($params);
    $data = $result->data['commerceApplyCoupon'];
    $this->assertEquals($data, $expected);
  }

  /**
   * Assert the removed coupon.
   *
   * @param array $variables
   *   The variables to pass to the query.
   * @param array $expected
   *   The expected values.
   */
  private function assertRemoveCoupon(array $variables, array $expected) {
    $query = <<<GQL
    mutation removeCoupon(\$order: ID, \$couponCode: String!) {
      commerceRemoveCoupon(
        couponCode: \$couponCode
        order: \$order
      ) {
         order {
          id
          totalPrice {
            number
          }
        }
        errors
      }
    }
    GQL;

    $params = OperationParams::create(
      [
        'query' => $query,
        'variables' => $variables,
      ]
    );
    $result = $this->server->executeOperation($params);
    $data = $result->data['commerceRemoveCoupon'];
    $this->assertEquals($data, $expected);
  }

}
