<?php

namespace Drupal\graphql_commerce\Plugin\GraphQL\DataProducer;

use Drupal\commerce_cart\CartProviderInterface;
use Drupal\commerce_checkout\CheckoutOrderManagerInterface;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_store\Entity\StoreInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Url;
use Drupal\graphql\GraphQL\Execution\FieldContext;
use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;
use Drupal\graphql_core_schema\GraphQL\Buffers\SubRequestBuffer;
use Drupal\user\UserInterface;
use GraphQL\Deferred;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * The checkout producer.
 *
 * @DataProducer(
 *   id = "commerce_checkout",
 *   name = @Translation("Commerce: Checkout"),
 *   description = @Translation("The checkout process."),
 *   consumes = {
 *     "order" = @ContextDefinition("entity:commerce_order",
 *       label = @Translation("The commerce order."),
 *       required = FALSE,
 *     ),
 *     "orderType" = @ContextDefinition("string",
 *       label = @Translation("The order type."),
 *       required = FALSE,
 *     ),
 *     "requestedStep" = @ContextDefinition("string",
 *       label = @Translation("The requested checkout step."),
 *       required = FALSE,
 *     ),
 *     "store" = @ContextDefinition("entity:commerce_store",
 *       label = @Translation("The store."),
 *       required = FALSE,
 *     ),
 *     "account" = @ContextDefinition("entity:user",
 *       label = @Translation("The user the cart belongs to."),
 *       required = FALSE,
 *     ),
 *   }
 * )
 */
class Checkout extends DataProducerPluginBase implements ContainerFactoryPluginInterface {

  /**
   * The checkout order manager.
   *
   * @var \Drupal\commerce_checkout\CheckoutOrderManagerInterface
   */
  protected $checkoutOrderManager;

  /**
   * The cart provider.
   *
   * @var \Drupal\commerce_cart\CartProviderInterface
   */
  protected $cartProvider;

  /**
   * The subrequest buffer.
   *
   * @var \Drupal\graphql_core_schema\GraphQL\Buffers\SubRequestBuffer
   */
  protected $subRequestBuffer;

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $pluginId,
    $pluginDefinition,
  ) {
    return new static(
      $configuration,
      $pluginId,
      $pluginDefinition,
      $container->get('commerce_checkout.checkout_order_manager'),
      $container->get('commerce_cart.cart_provider'),
      $container->get('graphql_core_schema.buffer.subrequest'),
    );
  }

  /**
   * The constructor.
   *
   * @param array $configuration
   *   The plugin configuration array.
   * @param string $pluginId
   *   The plugin id.
   * @param mixed $pluginDefinition
   *   The plugin definition.
   * @param \Drupal\commerce_checkout\CheckoutOrderManagerInterface $checkout_order_manager
   *   The checkout order manager.
   * @param \Drupal\commerce_cart\CartProviderInterface $cart_provider
   *   The cart provider.
   * @param \Drupal\graphql_core_schema\GraphQL\Buffers\SubRequestBuffer $subRequestBuffer
   *   The subrequest buffer.
   */
  public function __construct(
    array $configuration,
    $pluginId,
    $pluginDefinition,
    CheckoutOrderManagerInterface $checkout_order_manager,
    CartProviderInterface $cart_provider,
    SubRequestBuffer $subRequestBuffer,
  ) {
    parent::__construct($configuration, $pluginId, $pluginDefinition);
    $this->checkoutOrderManager = $checkout_order_manager;
    $this->cartProvider = $cart_provider;
    $this->subRequestBuffer = $subRequestBuffer;
  }

  /**
   * Advance the order to a checkout step.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   The order.
   * @param string $orderType
   *   The order type.
   * @param string $requestedStep
   *   The requested checkout step. If it's not available, the closest available
   *   step will be returned. The order's checkout step will not be altered.
   * @param \Drupal\commerce_store\Entity\StoreInterface $store
   *   The store to look a cart for..
   * @param \Drupal\user\UserInterface $account
   *   The account interface who the cart belongs to.
   * @param \Drupal\graphql\GraphQL\Execution\FieldContext $fieldContext
   *   The field Context.
   *
   * @return \GraphQL\Deferred
   *   The response.
   */
  public function resolve(?OrderInterface $order, ?string $orderType, ?string $requestedStep = NULL, ?StoreInterface $store = NULL, ?UserInterface $account = NULL, ?FieldContext $fieldContext = NULL): ?Deferred {
    if ($order && $order->get('cart') === FALSE) {
      return [];
    }
    if (!$order) {
      if (!$orderType) {
        $orderType = 'default';
      }
      $order = $this->cartProvider->getCart($orderType, $store, $account);
      if (!$order) {
        $order = $this->cartProvider->createCart($orderType, $store, $account);
      }
    }
    $url = Url::fromRoute(
      'commerce_checkout.form', [
        'commerce_order' => $order->id(),
      ]
    );

    $resolve = $this->subRequestBuffer->add(
      $url, function () use ($order, $requestedStep, $fieldContext) {
        $stepId = $this->checkoutOrderManager->getCheckoutStepId($order, $requestedStep);
        $checkoutFlow = $this->checkoutOrderManager->getCheckoutFlow($order);
        $checkoutFlowPlugin = $checkoutFlow->getPlugin();
        $fieldContext->setContextValue('commerce_checkout_step', $stepId);
        return $checkoutFlowPlugin;
      }
    );

    return new Deferred(
      function () use ($resolve) {
        $response = $resolve();
        return $response;
      }
    );
  }

}
