<?php

declare(strict_types=1);

namespace Drupal\Tests\graphql_commerce\Kernel\DataProducer;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_product\Entity\ProductVariationInterface;
use Drupal\commerce_store\Entity\StoreInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\graphql_commerce\Wrapper\CommerceOrderResponse;
use Drupal\Tests\graphql\Traits\DataProducerExecutionTrait;
use Drupal\Tests\graphql_commerce\Kernel\GraphQLCommerceOrderKernelTestBase;

/**
 * Tests the add_to_cart data producer.
 *
 * @coversDefaultClass \Drupal\graphql_commerce\Plugin\GraphQL\DataProducer\AddToCart
 * @group graphql_commerce
 */
class AddToCartTest extends GraphQLCommerceOrderKernelTestBase {

  use DataProducerExecutionTrait;

  /**
   * Tests adding items to the cart.
   *
   * @covers ::resolve
   */
  public function testResolveAddToCart(): void {
    // Try a non-existing order type.
    $added_quantity = 1;
    $response = $this->resolveAddToCart($this->variation, $added_quantity, 'xxx');
    $this->assertEmpty($response->getOrder());
    $this->assertEquals($response->getViolations()[0]['message'], 'This order type does not exist.');

    // User has no cart, add an item.
    $response = $this->resolveAddToCart($this->variation, $added_quantity, 'default');
    $order = $response->getOrder();
    $items = $order->getItems();
    $this->assertEquals($order->id(), 1);
    $this->assertEquals($items[0]->getQuantity(), $added_quantity);
    $this->assertEmpty($response->getViolations());

    // Add 2 more items of the same variation.
    $additional_quantity = 2;
    $this->resolveAddToCart($this->variation, $additional_quantity, '', $order);
    if (!empty($items)) {
      $actual_quantity = $items[0]->getQuantity();
    }
    $this->assertEquals($added_quantity + $additional_quantity, $actual_quantity);

    // Use another store.
    $another_store = $this->createStore('Another store', 'another@store.com', 'online', FALSE);
    $response = $this->resolveAddToCart($this->variation, $additional_quantity, 'default', NULL, $another_store, NULL);
    $another_order = $response->getOrder();
    $this->assertEquals($additional_quantity, $another_order->getItems()[0]->getQuantity());
  }

  /**
   * Execute the add to cart data producer.
   *
   * @param \Drupal\commerce_product\Entity\ProductVariationInterface $productVariation
   *   The product variation.
   * @param int|string $quantity
   *   The quantity of the product variation to add to the order.
   * @param string $orderType
   *   The order type. Used only if $order is NULL. If not given, 'default' is
   *   used.
   * @param \Drupal\commerce_order\Entity\OrderInterface|null $order
   *   The order to resolve the mutation on. Either this or all of the
   *   $orderType, $store and $account parameters are required.
   * @param \Drupal\commerce_store\Entity\StoreInterface|null $store
   *   The order's store. Used and only when $order is NULL. If not given, the
   *   default store is used.
   * @param \Drupal\Core\Session\AccountInterface|null $account
   *   The account the order should be assigned to. Used and required only
   *   when $order is NULL.
   *
   * @return \Drupal\graphql_commerce\Wrapper\CommerceOrderResponse
   *   The result of the mutation.
   */
  private function resolveAddToCart(ProductVariationInterface $productVariation, int|string $quantity, string $orderType, ?OrderInterface $order = NULL, ?StoreInterface $store = NULL, ?AccountInterface $account = NULL): CommerceOrderResponse {
    return $this->executeDataProducer(
      'commerce_add_to_cart', [
        'productVariation' => $productVariation,
        'quantity' => $quantity,
        'order' => $order,
        'orderType' => $orderType,
        'store' => $store,
        'account' => $account,
      ]
    );
  }

}
