<?php

declare(strict_types=1);

namespace Drupal\Tests\graphql_commerce\Kernel\DataProducer;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_store\Entity\StoreInterface;
use Drupal\Core\Routing\RouteObjectInterface;
use Drupal\Core\Url;
use Drupal\Tests\graphql_commerce\Kernel\GraphQLCommerceOrderKernelTestBase;
use Drupal\Tests\graphql\Traits\DataProducerExecutionTrait;
use Drupal\user\UserInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Session\Session;
use Symfony\Component\HttpFoundation\Session\Storage\MockArraySessionStorage;

/**
 * Tests the commerce_checkout data producer.
 *
 * @coversDefaultClass \Drupal\graphql_commerce\Plugin\GraphQL\DataProducer\Checkout
 * @group graphql_commerce
 */
class CheckoutTest extends GraphQLCommerceOrderKernelTestBase {

  use DataProducerExecutionTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'commerce_checkout',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installConfig('commerce_checkout');
  }

  /**
   * Fakes a request so that the current_route_match works.
   *
   * Copied over from CheckoutOrderManagerTest.
   *
   * @todo Remove this when CheckoutFlowBase stops using the route match.
   */
  protected function setupRequestWithOrderParameter(OrderInterface $order) {
    $url = Url::fromRoute('commerce_checkout.form', [
      'commerce_order' => $order->id(),
    ]);
    $route_provider = $this->container->get('router.route_provider');
    $route = $route_provider->getRouteByName($url->getRouteName());
    $request = Request::create($url->toString());
    $request->setSession(new Session(new MockArraySessionStorage()));
    $request->attributes->add([
      RouteObjectInterface::ROUTE_OBJECT => $route,
      'commerce_order' => $order,
    ]);
    $this->container->get('request_stack')->push($request);
  }

  /**
   * Tests the checkout producer.
   *
   * @covers ::resolve
   */
  public function testCheckout(): void {
    $cart = $this->cartProvider->createCart('default');
    $this->cartManager->addEntity($cart, $this->variation, 1);
    $this->setupRequestWithOrderParameter($cart);
    /** @var \Drupal\commerce_checkout\Plugin\Commerce\CheckoutFlow\CheckoutFlowInterface $response */
    $response = $this->resolveCheckout($cart);

    /** @var \Drupal\commerce_checkout\Entity\CheckoutFlowInterface $checkout_flow */
    $checkout_flow = $cart->get('checkout_flow')->entity;
    $plugin = $checkout_flow->getPlugin();
    $this->assertEquals($response, $plugin);
  }

  /**
   * Execute the add to cart data producer.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface|null $order
   *   The order. Either this or $orderType, $store and $account must be set.
   * @param string $orderType
   *   The order type. Used only if $order is NULL. If not given, 'default' is
   *   used.
   * @param string $requestedStep
   *   The requested checkout step.
   * @param \Drupal\commerce_store\Entity\StoreInterface|null $store
   *   The order's store. Used only when $order is NULL. If not given, the
   *   default store is used.
   * @param \Drupal\Core\Session\AccountInterface|null $account
   *   The account the order should be assigned to. Used only when $order is
   *   NULL.
   *
   * @return \Drupal\commerce_checkout\Entity\CheckoutFlowInterface
   *   The checkout flow.
   */
  private function resolveCheckout(?OrderInterface $order, ?string $orderType = NULL, ?string $requestedStep = NULL, ?StoreInterface $store = NULL, ?UserInterface $account = NULL) {
    return $this->executeDataProducer(
      'commerce_checkout', [
        'order' => $order,
        'orderType' => $orderType,
        'requestedStep' => $requestedStep,
        'store' => $store,
        'account' => $account,
      ]
    );
  }

}
