<?php

declare(strict_types=1);

namespace Drupal\Tests\graphql_commerce\Kernel\SchemaExtension;

use Drupal\graphql\Entity\ServerInterface;
use Drupal\Tests\graphql_commerce\Kernel\GraphQLCommerceOrderKernelTestBase;
use GraphQL\Server\OperationParams;

/**
 * Tests the commerce_add_to_cart extension.
 *
 * @group graphql_core_schema
 */
class AddToCartExtensionTest extends GraphQLCommerceOrderKernelTestBase {

  /**
   * The GraphQL server.
   */
  protected ServerInterface $server;

  /**
   * Test the Url type resolvers.
   */
  public function testAddToCartResolver(): void {
    $this->server = $this
      ->getCoreComposableServerBuilder()
      ->enableValueFields()
      ->enableExtension('commerce')
      ->enableEntityType('commerce_order', ['id', 'order_items'], ['default'])
      ->enableEntityType('commerce_product_variation', [], ['default'])
      ->enableEntityType('commerce_order_item', ['id', 'quantity', 'title'], ['default'])
      ->enableEntityType('user', ['uid'])
      ->createServer();

    $quantity = 3;
    // Try a nonexistent order type.
    $variables = [
      'quantity' => $quantity,
      'productVariation' => 1,
      'store' => '',
      'orderType' => 'xxx',
      'order' => '',
      'account' => '',
    ];
    $expected = [
      'order' => '',
      'errors' => [
        [
          'message' => 'This order type does not exist.',
        ],
      ],
    ];
    $this->assertAddToCart($variables, $expected);

    // Add something to the cart for real.
    $variables['orderType'] = 'default';
    $expected['errors'] = [];
    $expected['order'] = [
      'id' => 1,
      'orderItems' => [
        [
          'title' => $this->variation->label(),
          'quantity' => $quantity,
        ],
      ],
    ];
    $this->assertAddToCart($variables, $expected);
  }

  /**
   * Assert the resolved Url type.
   *
   * @param array $variables
   *   The variables to pass to the query.
   * @param array $expected
   *   The expected values.
   */
  private function assertAddToCart(array $variables, array $expected) {
    $query = <<<GQL
    mutation addToCart(\$order: ID, \$account: ID, \$orderType: String, \$productVariation: ID!, \$quantity: Int!, \$store: ID) {
      commerceAddToCart(
        quantity: \$quantity
        productVariation: \$productVariation
        store: \$store
        orderType: \$orderType
        order: \$order
        account: \$account
      ) {
        errors
        order {
          id
          orderItems {
            quantity
            title
          }
        }
      }
    }
    GQL;

    $params = OperationParams::create(
      [
        'query' => $query,
        'variables' => $variables,
      ]
    );
    $result = $this->server->executeOperation($params);
    $data = $result->data['commerceAddToCart'];
    $this->assertEquals($data, $expected);
  }

}
