<?php

declare(strict_types=1);

namespace Drupal\Tests\graphql_commerce\Kernel\SchemaExtension;

use Drupal\graphql\Entity\ServerInterface;
use Drupal\Tests\graphql_commerce\Kernel\GraphQLCommerceOrderKernelTestBase;
use GraphQL\Server\OperationParams;

/**
 * Tests the commerce_cart_change_quantity extension.
 *
 * @group graphql_core_schema
 */
class ChangeQuantityExtensionTest extends GraphQLCommerceOrderKernelTestBase {

  /**
   * The GraphQL server.
   */
  protected ServerInterface $server;

  /**
   * Test the Url type resolvers.
   */
  public function testEmptyCartResolver(): void {
    $this->server = $this
      ->getCoreComposableServerBuilder()
      ->enableValueFields()
      ->enableExtension('commerce')
      ->enableEntityType('commerce_order', ['id', 'order_items'], ['default'])
      ->enableEntityType('commerce_product_variation', [], ['default'])
      ->enableEntityType('commerce_order_item', ['id', 'quantity', 'title'], ['default'])
      ->createServer();

    $another_variation = $this->createEntity(
      'commerce_product_variation', [
        'type' => 'default',
        'sku' => $this->randomMachineName(),
        'price' => [
          'number' => 259,
          'currency_code' => 'USD',
        ],
        'product_id' => $this->product->id(),
        'title' => 'Some temporary title',
      ]
    );
    $cart = $this->cartProvider->createCart('default');
    $this->cartManager->addEntity($cart, $this->variation, 2);
    $item = $this->cartManager->addEntity($cart, $another_variation, 3);
    $new_quantity = 14;
    $variables = [
      'item' => $item->id(),
      'quantity' => $new_quantity,
    ];
    $expected = [
      'errors' => [],
      'order' => [
        'orderItems' => [
          [
            'title' => $this->variation->label(),
            'quantity' => 2,
          ],
          [
            'title' => $another_variation->label(),
            'quantity' => $new_quantity,
          ],
        ],
      ],
    ];
    $this->assertChangeQuantity($variables, $expected);
  }

  /**
   * Assert the resolved empty cart type.
   *
   * @param array $variables
   *   The variables to pass to the query.
   * @param array $expected
   *   The expected values.
   */
  private function assertChangeQuantity(array $variables, array $expected) {
    $query = <<<GQL
    mutation MyMutation(\$quantity: Int!, \$item: ID!) {
      commerceCartChangeQuantity(
        quantity: \$quantity,
        item: \$item
      ) {
        errors
        order {
          orderItems {
            title
            quantity
          }
        }
      }
    }
    GQL;

    $params = OperationParams::create(
      [
        'query' => $query,
        'variables' => $variables,
      ]
    );
    $result = $this->server->executeOperation($params);
    $data = $result->data['commerceCartChangeQuantity'];
    $this->assertEquals($expected, $data);
  }

}
