<?php

declare(strict_types=1);

namespace Drupal\Tests\graphql_commerce\Kernel\SchemaExtension;

use Drupal\graphql\Entity\ServerInterface;
use Drupal\Tests\graphql_commerce\Kernel\GraphQLCommerceOrderKernelTestBase;
use GraphQL\Server\OperationParams;

/**
 * Tests the commerce_checkout extension.
 *
 * @group graphql_core_schema
 */
class CheckoutExtensionTest extends GraphQLCommerceOrderKernelTestBase {

  /**
   * The GraphQL server.
   */
  protected ServerInterface $server;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'commerce_checkout',
    'graphql_form_schema',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installConfig('commerce_checkout');
  }

  /**
   * Test the checkout schema extension.
   */
  public function testCheckoutResolver(): void {
    $cart = $this->cartProvider->createCart('default');
    $this->cartManager->addEntity($cart, $this->variation, 1);
    $storage = $this->entityTypeManager->getStorage('commerce_order');

    $this->server = $this
      ->getCoreComposableServerBuilder()
      ->enableValueFields()
      ->enableExtension('commerce')
      ->enableExtension('commerce_checkout')
      ->enableExtension('form')
      ->enableEntityType('commerce_order', ['id', 'order_items'], ['default'])
      ->enableEntityType('commerce_product_variation', [], ['default'])
      ->enableEntityType('commerce_order_item', ['id', 'quantity', 'title'], ['default'])
      ->enableEntityType('user', ['uid'])
      ->createServer();
    // Try a nonexistent order type.
    $variables = [
      'order' => 1,
    ];
    $expected = [
      'nextStep' => [
        'id' => 'order_information',
      ],
      'currentStep' => [
        'id' => 'login',
      ],
      'order' => [
        'id' => $cart->id(),
      ],
    ];
    $this->assertCheckout($variables, $expected);

    // Proceed to review and see available steps.
    // Avoid an order version mismatch error.
    $cart = $storage->load($cart->id());
    $cart->set('checkout_step', 'review')->save();
    $expected = [
      'nextStep' => [
        'id' => 'complete',
      ],
      'currentStep' => [
        'id' => 'review',
      ],
      'order' => [
        'id' => $cart->id(),
      ],
    ];
    $this->assertCheckout($variables, $expected);

    // A previous step should be available.
    $variables['requestedStep'] = 'order_information';
    $expected['currentStep']['id'] = 'order_information';
    $expected['nextStep']['id'] = 'review';
    $this->assertCheckout($variables, $expected);

    // A following step should not be available.
    $variables['requestedStep'] = 'complete';
    $expected['currentStep']['id'] = 'review';
    $expected['nextStep']['id'] = 'complete';
    $this->assertCheckout($variables, $expected);
  }

  /**
   * Assert the checkout.
   *
   * @param array $variables
   *   The variables to pass to the query.
   * @param array $expected
   *   The expected values.
   */
  private function assertCheckout(array $variables, array $expected) {
    $variables['requestedStep'] ??= $expected['currentStep']['id'];
    $query = <<<GQL
    query checkout(\$order: ID, \$requestedStep: String) {
      commerceCheckout(order: \$order, requestedStep: \$requestedStep) {
        nextStep {
          id
        }
        currentStep {
          id
        }
        order {
          id
        }
      }
    }
    GQL;

    $params = OperationParams::create(
      [
        'query' => $query,
        'variables' => $variables,
      ]
    );
    $result = $this->server->executeOperation($params);
    $data = $result->data['commerceCheckout'];
    $this->assertEquals($expected, $data);
  }

}
