<?php

declare(strict_types=1);

namespace Drupal\Tests\graphql_commerce\Kernel\SchemaExtension;

use Drupal\graphql\Entity\ServerInterface;
use Drupal\Tests\graphql_commerce\Kernel\GraphQLCommerceOrderKernelTestBase;
use GraphQL\Server\OperationParams;

/**
 * Tests the commerce_empty_cart extension.
 *
 * @group graphql_core_schema
 */
class EmptyCartExtensionTest extends GraphQLCommerceOrderKernelTestBase {

  /**
   * The GraphQL server.
   */
  protected ServerInterface $server;

  /**
   * Test the Url type resolvers.
   */
  public function testEmptyCartResolver(): void {
    $this->server = $this
      ->getCoreComposableServerBuilder()
      ->enableValueFields()
      ->enableExtension('commerce')
      ->enableEntityType('commerce_order', ['id', 'order_items'], ['default'])
      ->enableEntityType('commerce_product_variation', [], ['default'])
      ->enableEntityType('commerce_order_item', ['id', 'quantity', 'purchased_entity'], ['default'])
      ->createServer();

    $another_variation = $this->createEntity(
      'commerce_product_variation', [
        'type' => 'default',
        'sku' => $this->randomMachineName(),
        'price' => [
          'number' => 259,
          'currency_code' => 'USD',
        ],
        'title' => 'Some temporary title',
        'product_id' => $this->product->id(),
      ]
    );

    $cart = $this->cartProvider->createCart('default');
    $this->cartManager->addEntity($cart, $this->variation, 2);
    $this->cartManager->addEntity($cart, $another_variation, 3);

    $variables = [
      'order' => 1,
    ];
    $expected = [
      'errors' => [],
      'order' => [
        'id' => 1,
        'orderItems' => [],
      ],
    ];
    $this->assertEmptyCart($variables, $expected);
  }

  /**
   * Assert the resolved empty cart type.
   *
   * @param array $variables
   *   The variables to pass to the query.
   * @param array $expected
   *   The expected values.
   */
  private function assertEmptyCart(array $variables, array $expected) {
    $query = <<<GQL
    mutation emptyCart(\$order: ID!) {
      commerceEmptyCart(
        order: \$order
      ) {
        errors
        order {
          id
          orderItems {
            id
          }
        }
      }
    }
    GQL;

    $params = OperationParams::create(
      [
        'query' => $query,
        'variables' => $variables,
      ]
    );
    $result = $this->server->executeOperation($params);
    $data = $result->data['commerceEmptyCart'];
    $this->assertEquals($data, $expected);
  }

}
