<?php

declare(strict_types=1);

namespace Drupal\Tests\graphql_commerce\Kernel\SchemaExtension;

use Drupal\graphql\Entity\ServerInterface;
use Drupal\Tests\graphql_commerce\Kernel\GraphQLCommerceOrderKernelTestBase;
use GraphQL\Server\OperationParams;

/**
 * Tests the commerce_get_carts extension.
 *
 * @group graphql_core_schema
 */
class GetCartsExtensionTest extends GraphQLCommerceOrderKernelTestBase {

  /**
   * The GraphQL server.
   */
  protected ServerInterface $server;

  /**
   * Test the Url type resolvers.
   */
  public function testGetCartsResolver(): void {
    $this->server = $this
      ->getCoreComposableServerBuilder()
      ->enableValueFields()
      ->enableExtension('commerce')
      ->enableEntityType('commerce_order', ['id'], ['default'])
      ->enableEntityType('commerce_product_variation')
      ->enableEntityType('commerce_product')
      ->enableEntityType('commerce_store', ['id'], ['default'])
      ->enableEntityType('user', ['uid'])
      ->createServer();

    // Try a nonexistent order type.
    $variables = [
      'orderType' => 'xxx',
    ];
    $expected = [];
    $this->assertGetCarts($variables, $expected);

    $order_type = 'default';
    $cart = $this->cartProvider->createCart($order_type, $this->store, $this->user);
    $this->cartManager->addEntity($cart, $this->variation);

    $another_store = $this->createStore('Another store', 'another@store.com', 'online', FALSE);
    $another_cart = $this->cartProvider->createCart($order_type, $another_store, $this->user);
    $this->cartManager->addEntity($another_cart, $this->variation);

    $another_user = $this->createUser();
    $third_cart = $this->cartProvider->createCart($order_type, $another_store, $another_user);
    $this->cartManager->addEntity($third_cart, $this->variation);

    $fourth_cart = $this->cartProvider->createCart($order_type, $this->store, $another_user);
    $this->cartManager->addEntity($fourth_cart, $this->variation);

    $variables['orderType'] = $order_type;
    $expected = [
      ['id' => 1],
      ['id' => 2],
      ['id' => 3],
      ['id' => 4],
    ];
    $this->assertGetCarts($variables, $expected);

    $variables['accountId'] = $this->user->id();
    $expected = [
      ['id' => 1],
      ['id' => 2],
    ];
    $this->assertGetCarts($variables, $expected);

    $variables['accountId'] = $another_user->id();
    $expected = [
      ['id' => 3],
      ['id' => 4],
    ];
    $this->assertGetCarts($variables, $expected);

    $variables['storeId'] = $this->store->id();
    $expected = [
      ['id' => 4],
    ];
    $this->assertGetCarts($variables, $expected);

    $variables['storeId'] = $another_store->id();
    $expected = [
      ['id' => 3],
    ];
    $this->assertGetCarts($variables, $expected);

    $variables['accountId'] = $this->user->id();
    $expected = [
      ['id' => 2],
    ];
    $this->assertGetCarts($variables, $expected);

    $variables['storeId'] = $this->store->id();
    $expected = [
      ['id' => 1],
    ];
    $this->assertGetCarts($variables, $expected);

    unset($variables['accountId']);
    $expected = [
      ['id' => 1],
      ['id' => 4],
    ];
    $this->assertGetCarts($variables, $expected);

    $variables['storeId'] = $another_store->id();
    $expected = [
      ['id' => 2],
      ['id' => 3],
    ];
    $this->assertGetCarts($variables, $expected);
  }

  /**
   * Assert the resolved get carts.
   *
   * @param array $variables
   *   The variables to pass to the query.
   * @param array $expected
   *   The expected values.
   */
  private function assertGetCarts(array $variables, array $expected) {
    $query = <<<GQL
    query getCarts(\$orderType: String, \$accountId: ID, \$storeId: ID) {
      commerceGetCarts(
        orderType: \$orderType,  
        accountId: \$accountId, 
        storeId: \$storeId) {
          id
        }
      }
    GQL;

    $params = OperationParams::create(
      [
        'query' => $query,
        'variables' => $variables,
      ]
    );
    $result = $this->server->executeOperation($params);
    $data = $result->data['commerceGetCarts'];
    $this->assertEquals($data, $expected);
  }

}
