<?php

declare(strict_types=1);

namespace Drupal\Tests\graphql_commerce\Kernel\SchemaExtension;

use Drupal\Tests\graphql_commerce\Kernel\ProductAttributesTestBase;
use GraphQL\Server\OperationParams;

/**
 * Tests the commerce_attributes extension.
 *
 * @group graphql_core_schema
 */
class ProductAttributesExtensionTest extends ProductAttributesTestBase {

  /**
   * Test the product attributes.
   */
  public function testComputedAvailabilityExtension(): void {
    $this->variation = $this->entityTypeManager->getStorage('commerce_product_variation')->load($this->variation->id());
    $this->variation->set('attribute_color', $this->colorAttributes['blue']->id(),);
    $this->variation->save();

    $this->server = $this
      ->getCoreComposableServerBuilder()
      ->enableValueFields()
      ->enableExtension('commerce_attribute')
      ->enableExtension('entity_query')
      ->enableBaseEntityField('label')
      ->enableBaseEntityField('accessCheck')

      ->enableEntityType('commerce_product_variation', ['variation_id', 'attribute_color'], ['default'])
      ->enableEntityType('commerce_product_attribute_value', ['attribute', 'attribute_value_id', 'name'], ['color'])
      ->enableEntityType('commerce_product_attribute', ['variation_id'], ['default'])
      ->enableEntityType('user', ['uid'])
      ->createServer();

    $user = $this->setUpCurrentUser();
    // entityById does not have a "skip access checks" switch.
    // 'access content' is needed to get results in the attributeColor field.
    $role = $this->createRole(['view commerce_product', 'access content']);
    $user->addRole($role);
    $this->setCurrentUser($user);

    $variables = [
      'productVariation' => $this->variation->id(),
    ];
    $expected = [
      'id' => $this->variation->id(),
      'attributes' => [
        [
          'fieldName' => 'attribute_color',
          'label' => 'Color',
          'id' => 'color',
          'values' => [
            [
              'id' => 2,
              'label' => 'Blue',
            ],
          ],
          'elementType' => 'select',
          'required' => TRUE,
        ],
      ],
      'attributeColor' => [
        'id' => 2,
        'label' => 'Blue',
      ],
    ];
    $this->assertProductAttributes($variables, $expected);
  }

  /**
   * Assert the product attributes.
   *
   * @param array $variables
   *   The variables to pass to the query.
   * @param array $expected
   *   The expected values.
   */
  private function assertProductAttributes(array $variables, array $expected) {
    $query = <<<GQL
      query productAttributes(\$productVariation: ID!) {
        entityById(entityType: COMMERCE_PRODUCT_VARIATION, id: \$productVariation) {
          ... on CommerceProductVariationDefault {
            id
            attributes {
              fieldName
              label
              id
              values {
                id
                label
              }
              elementType
              required
            }
            attributeColor {
              id
              label
            }
          }
        }
      }
    GQL;

    $params = OperationParams::create(
      [
        'query' => $query,
        'variables' => $variables,
      ]
    );
    $result = $this->server->executeOperation($params);
    $data = $result->data['entityById'];
    $this->assertEquals($data, $expected);
  }

}
