<?php

namespace Drupal\graphql_commerce_promotion\Plugin\GraphQL\DataProducer;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\Core\Render\RenderContext;
use Drupal\graphql\GraphQL\Execution\FieldContext;
use Drupal\graphql_commerce\Plugin\GraphQL\CommerceCartDataProducerBase;
use Drupal\graphql_commerce\Wrapper\CommerceOrderResponse;

/**
 * Applies a coupon on the given order.
 *
 * @DataProducer(
 *   id = "commerce_promotion_apply_coupon",
 *   name = @Translation("Commerce Promotion: Apply Coupon"),
 *   description = @Translation("Applies a coupon on the given order."),
 *   produces = @ContextDefinition("any",
 *     label = @Translation("The mutation result.")
 *   ),
 *   consumes = {
 *     "couponCode" = @ContextDefinition("string",
 *       label = @Translation("The coupon to apply."),
 *     ),
 *     "order" = @ContextDefinition("entity:commerce_order",
 *       label = @Translation("The commerce order."),
 *       required = FALSE,
 *     )
 *   }
 * )
 */
class ApplyCoupon extends CommerceCartDataProducerBase {

  /**
   * Resolves the coupon code for an order and applies it if applicable.
   *
   * @param string $couponCode
   *   The coupon code to apply.
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   The order entity to apply the coupon code to.
   * @param \Drupal\graphql\GraphQL\Execution\FieldContext $field
   *   The field context object.
   *
   * @return \Drupal\graphql_commerce\Wrapper\CommerceOrderResponse
   *   The result of applying the coupon code.
   */
  public function resolve(string $couponCode, ?OrderInterface $order, ?FieldContext $field): CommerceOrderResponse {
    $context = new RenderContext();

    $result = $this->renderer->executeInRenderContext(
      $context, function () use ($order, $couponCode) {
        if (!$order) {
          $order = $this->cartProvider->getCart('default');
          if (!$order) {
            $order = $this->cartProvider->createCart('default');
          }
        }
        $result = new CommerceOrderResponse();
        $result->setOrder($order);

        /** @var \Drupal\commerce_promotion\CouponStorageInterface $couponStorage */
        $couponStorage = $this->entityTypeManager->getStorage('commerce_promotion_coupon');
        $coupon = $couponStorage->loadEnabledByCode($couponCode);

          // Check if coupon exists.
        if (empty($coupon)) {
          $result->addViolation($this->t('The provided coupon code is invalid.'));
          return $result;
        }

          // Check if the coupon has already been applied.
        foreach ($order->get('coupons') as $item) {
          if ($item->target_id == $coupon->id()) {
            $result->addViolation($this->t('The provided coupon has already been applied.'));
            return $result;
          }
        }

          // Check if the coupon is available.
        if (!$coupon->available($order)) {
          $result->addViolation($this->t('The provided coupon code is not available. It may have expired or already been used.'));
          return $result;
        }

          // Check if the coupon applies to this order.
        if (!$coupon->getPromotion()->applies($order)) {
          $result->addViolation($this->t('The provided coupon code cannot be applied to your order.'));
          return $result;
        }

        // Add coupon and save order.
        $order->get('coupons')->appendItem($coupon->id());
        $order->save();

        return $result;
      }
    );

    $field->addCacheableDependency($context);
    return $result;
  }

}
