<?php

namespace Drupal\graphql_commerce\Plugin\GraphQL\DataProducer;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_product\Entity\ProductVariationInterface;
use Drupal\commerce_store\Entity\StoreInterface;
use Drupal\Core\Render\RenderContext;
use Drupal\Core\Session\AccountInterface;
use Drupal\graphql\GraphQL\Execution\FieldContext;
use Drupal\graphql_commerce\Plugin\GraphQL\CommerceCartDataProducerBase;
use Drupal\graphql_commerce\Wrapper\CommerceOrderResponse;

/**
 * The add to cart producer.
 *
 * @DataProducer(
 *   id = "commerce_add_to_cart",
 *   name = @Translation("Commerce: Add to Cart"),
 *   description = @Translation("Add a product to the cart."),
 *   produces = @ContextDefinition("any",
 *     label = @Translation("The mutation result.")
 *   ),
 *   consumes = {
 *     "productVariation" = @ContextDefinition("any",
 *       label = @Translation("The product variation to add to the cart."),
 *       required = TRUE,
 *     ),
 *     "quantity" = @ContextDefinition("any",
 *       label = @Translation("The quantity to add to the cart."),
 *     ),
 *     "order" = @ContextDefinition("entity:commerce_order",
 *       label = @Translation("The commerce order."),
 *       required = FALSE,
 *     ),
 *     "orderType" = @ContextDefinition("string",
 *       label = @Translation("The order type."),
 *       required = FALSE,
 *     ),
 *     "store" = @ContextDefinition("entity:commerce_store",
 *       label = @Translation("The store."),
 *       required = FALSE,
 *     ),
 *     "account" = @ContextDefinition("entity:user",
 *       label = @Translation("The user the cart belongs to."),
 *       required = FALSE,
 *     ),
 *   }
 * )
 */
class AddToCart extends CommerceCartDataProducerBase {

  /**
   * Resolves a mutation on an order.
   *
   * @param \Drupal\commerce_product\Entity\ProductVariationInterface $productVariation
   *   The product variation.
   * @param int|string $quantity
   *   The quantity of the product variation to add to the order.
   * @param \Drupal\commerce_order\Entity\OrderInterface|null $order
   *   The order to resolve the mutation on. Either this or all of the
   *   $orderType, $store and $account parameters are required.
   * @param string $orderType
   *   The order type. Used only if $order is NULL. If not given, 'default' is
   *   used.
   * @param \Drupal\commerce_store\Entity\StoreInterface|null $store
   *   The order's store. Used and only when $order is NULL. If not given, the
   *   default store is used.
   * @param \Drupal\Core\Session\AccountInterface|null $account
   *   The account the order should be assigned to. Used and required only
   *   when $order is NULL.
   * @param \Drupal\graphql\GraphQL\Execution\FieldContext $field
   *   The field context.
   *
   * @return \Drupal\graphql_commerce\Wrapper\CommerceOrderResponse
   *   The result of the mutation.
   */
  public function resolve(ProductVariationInterface $productVariation, int|string $quantity, ?OrderInterface $order = NULL, ?string $orderType = NULL, ?StoreInterface $store = NULL, ?AccountInterface $account = NULL, ?FieldContext $field = NULL): CommerceOrderResponse {
    $response = new CommerceOrderResponse($order);
    if (!$order) {
      $this->validateOrderType($orderType, $response);
      if ($response->getViolations()) {
        return $response;
      }

      $order = $this->cartProvider->getCart($orderType, $store, $account);
      if (!$order) {
        $order = $this->cartProvider->createCart($orderType, $store, $account);
      }
      $response = new CommerceOrderResponse($order);
    }

    $context = new RenderContext();
    $this->renderer->executeInRenderContext(
      $context, function () use ($order, $productVariation, $quantity, $response) {
        // Load the 'combine' setting from the AddToCart formatter on the view
        // display for the bundle.
        /** @var \Drupal\Core\Entity\EntityDisplayRepositoryInterface $entityDisplayRepository */
        $entityDisplayRepository = \Drupal::service('entity_display.repository');
        $bundle = $productVariation->getProduct()?->bundle();
        $combine = TRUE;
        if ($bundle) {
          $viewDisplay = $entityDisplayRepository->getViewDisplay('commerce_product', $bundle);
          $combine = $viewDisplay->get('content')['variations']['settings']['combine'] ?? TRUE;
        }
        if (!$this->addStockViolationsToResult($productVariation, $quantity, $response)) {
          $this->cartManager->addEntity($order, $productVariation, $quantity, $combine);
        }
      }
    );

    $field->addCacheableDependency($context);
    return $response;
  }

}
