<?php

namespace Drupal\graphql_commerce\Plugin\GraphQL\DataProducer;

use Drupal\commerce_order\Entity\OrderItemInterface;
use Drupal\Core\Render\RenderContext;
use Drupal\graphql\GraphQL\Execution\FieldContext;
use Drupal\graphql_commerce\Plugin\GraphQL\CommerceCartDataProducerBase;
use Drupal\graphql_commerce\Wrapper\CommerceOrderResponse;

/**
 * The change quantity producer.
 *
 * @DataProducer(
 *   id = "commerce_cart_change_quantity",
 *   name = @Translation("Commerce Cart: Change Quantity"),
 *   description = @Translation("Change the quantity of an order item."),
 *   produces = @ContextDefinition("any",
 *     label = @Translation("The mutation result.")
 *   ),
 *   consumes = {
 *     "item" = @ContextDefinition("entity:commerce_order_item",
 *       label = @Translation("The commerce order item."),
 *     ),
 *     "quantity" = @ContextDefinition("integer",
 *       label = @Translation("The new quantity."),
 *     )
 *   }
 * )
 */
class ChangeQuantity extends CommerceCartDataProducerBase {

  /**
   * The resolver.
   *
   * @param \Drupal\commerce_order\Entity\OrderItemInterface $item
   *   The order item.
   * @param int $quantity
   *   The new quantity.
   * @param \Drupal\graphql\GraphQL\Execution\FieldContext $field
   *   The field context.
   *
   * @return \Drupal\graphql_commerce\Wrapper\CommerceOrderResponse
   *   The result of the mutation.
   */
  public function resolve(OrderItemInterface $item, int $quantity, FieldContext $field): CommerceOrderResponse {
    $order = $item->getOrder();
    $result = new CommerceOrderResponse($order);

    $context = new RenderContext();
    $this->renderer->executeInRenderContext(
      $context, function () use ($order, $item, $quantity, $result) {
        if ($quantity === 0) {
          $this->cartManager->removeOrderItem($order, $item, save_cart: TRUE);
          return $result;
        }
        $item->setQuantity($quantity);
        // These could be related to availability or other sources.
        $violations = $item->validate();
        if ($violations->count() === 0) {
          $quantityDelta = $this->getQuantityDelta($quantity, $item->getPurchasedEntity());
          if (!$this->addStockViolationsToResult($item->getPurchasedEntity(), $quantityDelta, $result)) {
            $this->cartManager->updateOrderItem($order, $item, save_cart: TRUE);
          }
        }
        else {
          foreach ($violations as $violation) {
            $result->addViolation($violation->getMessage());
          }
        }
      }
    );
    $field->addCacheableDependency($context);
    return $result;
  }

}
