<?php

declare(strict_types=1);

namespace Drupal\graphql_commerce\Plugin\GraphQL\DataProducer;

use Drupal\commerce_store\Entity\StoreInterface;
use Drupal\Core\Render\RenderContext;
use Drupal\Core\Session\AccountInterface;
use Drupal\graphql\GraphQL\Execution\FieldContext;
use Drupal\graphql_commerce\Plugin\GraphQL\CommerceCartDataProducerBase;
use Drupal\graphql_commerce\Wrapper\CommerceOrderResponse;

/**
 * The get cart data producer.
 *
 * @DataProducer(
 *   id = "commerce_get_cart",
 *   name = @Translation("Commerce: Get Cart"),
 *   description = @Translation("Get the cart for the current session."),
 *   produces = @ContextDefinition("entity:commerce_order",
 *     label = @Translation("The commerce cart."),
 *   ),
 *   consumes = {
 *     "create" = @ContextDefinition("boolean",
 *       label = @Translation("If a new cart should be created."),
 *     ),
 *     "orderType" = @ContextDefinition("string",
 *       label = @Translation("The order bundle."),
 *     ),
 *     "store" = @ContextDefinition("entity:commerce_store",
 *       label = @Translation("The store."),
 *       required = FALSE,
 *     ),
 *     "account" = @ContextDefinition("entity:user",
 *       label = @Translation("The account for which the cart should be created."),
 *       required = FALSE,
 *     ),
 *   }
 * )
 */
class GetCart extends CommerceCartDataProducerBase {

  /**
   * Resolves the cart for the current session.
   *
   * @param bool $create
   *   Whether a new cart should be created if none exists.
   * @param string $orderType
   *   The order bundle.
   * @param \Drupal\commerce_store\Entity\StoreInterface|null $store
   *   The store to create the cart for.
   * @param \Drupal\Core\Session\AccountInterface|null $account
   *   The user to create the cart for.
   * @param \Drupal\graphql\GraphQL\Execution\FieldContext $field
   *   The field context.
   *
   * @return \Drupal\graphql_commerce\Wrapper\CommerceOrderResponse
   *   The order response.
   */
  public function resolve(bool $create, string $orderType, ?StoreInterface $store = NULL, ?AccountInterface $account = NULL, ?FieldContext $field = NULL): CommerceOrderResponse {
    $response = new CommerceOrderResponse();
    $this->validateOrderType($orderType, $response);
    if ($response->getViolations()) {
      return $response;
    }
    $context = new RenderContext();
    $cart = NULL;
    $this->renderer->executeInRenderContext(
      $context, function () use (&$cart, $orderType, $store, $account, $create) {
        $cart = $this->cartProvider->getCart($orderType, $store, $account);
        // Create a cart if requested.
        if (!$cart && $create) {
          $cart = $this->cartProvider->createCart($orderType, $store, $account);
        }
      }
    );

    $field->addCacheableDependency($cart);
    $field->addCacheableDependency($context);
    if ($cart) {
      if (!$cart->access('view', $account)) {
        $response->addViolation($this->t('This user has no access to the cart.'));
      }
      $response->setOrder($cart);
    }

    return $response;
  }

}
