<?php

namespace Drupal\graphql_commerce\Plugin\GraphQL\DataProducer;

use Drupal\commerce_order\Entity\OrderItemInterface;
use Drupal\Core\Render\RenderContext;
use Drupal\graphql\GraphQL\Execution\FieldContext;
use Drupal\graphql_commerce\Plugin\GraphQL\CommerceCartDataProducerBase;
use Drupal\graphql_commerce\Wrapper\CommerceOrderResponse;

/**
 * Data producer to remove item from the cart.
 *
 * @DataProducer(
 *   id = "commerce_remove_from_cart",
 *   name = @Translation("Commerce: Remove from Cart"),
 *   description = @Translation("Removes an order item from the cart."),
 *   produces = @ContextDefinition("any",
 *     label = @Translation("The mutation response."),
 *   ),
 *   consumes = {
 *     "item" = @ContextDefinition("entity:commerce_order_item",
 *       label = @Translation("The commerce order item to remove."),
 *     ),
 *   }
 * )
 */
class RemoveFromCart extends CommerceCartDataProducerBase {

  /**
   * Resolves the given order item ID and removes it from the order.
   *
   * @param \Drupal\commerce_order\Entity\OrderItemInterface $item
   *   The order object for which to remove the order item.
   * @param \Drupal\graphql\GraphQL\Execution\FieldContext $field
   *   The field context object.
   *
   * @return \Drupal\graphql_commerce\Wrapper\CommerceOrderResponse
   *   The response of the mutation operation.
   */
  public function resolve(OrderItemInterface $item, FieldContext $field) {
    $order = $item->getOrder();
    $result = new CommerceOrderResponse($order);
    $context = new RenderContext();
    $this->renderer->executeInRenderContext(
      $context, function () use ($order, $item) {
        if ($order) {
          $this->cartManager->removeOrderItem($order, $item);
        }
        // This might be an orphan item.
        else {
          $item->delete();
        }
      }
    );

    $field->addCacheableDependency($context);
    return $result;
  }

}
