<?php

declare(strict_types=1);

namespace Drupal\Tests\graphql_commerce\Kernel\DataProducer;

use Drupal\commerce_order\Entity\OrderItemInterface;
use Drupal\graphql_commerce\Wrapper\CommerceOrderResponse;
use Drupal\Tests\graphql\Traits\DataProducerExecutionTrait;
use Drupal\Tests\graphql_commerce\Kernel\GraphQLCommerceOrderKernelTestBase;

/**
 * Tests the commerce_cart_change_quantity data producer.
 *
 * @coversDefaultClass \Drupal\graphql_commerce\Plugin\GraphQL\DataProducer\ChangeQuantity
 * @group graphql_commerce
 */
class ChangeQuantityTest extends GraphQLCommerceOrderKernelTestBase {

  use DataProducerExecutionTrait;

  /**
   * Tests removing item from the cart.
   *
   * @covers ::resolve
   */
  public function testResolveChangeQuantity(): void {
    $cart = $this->cartProvider->createCart('default');
    $item = $this->cartManager->addEntity($cart, $this->variation, 2);
    $another_variation = $this->createEntity(
      'commerce_product_variation', [
        'type' => 'default',
        'sku' => $this->randomMachineName(),
        'price' => [
          'number' => 259,
          'currency_code' => 'USD',
        ],
        'title' => 'Some temporary title',
        'product_id' => $this->product->id(),
      ]
    );
    $item = $this->cartManager->addEntity($cart, $another_variation, 21);

    $new_quantity = 11;
    $response = $this->resolveChangeQuantity($item, $new_quantity);
    $this->assertNotEmpty($response->getOrder());
    $this->assertEquals($item->getQuantity(), $new_quantity);
  }

  /**
   * Execute the remove from cart data producer.
   *
   * Resolves the given order item ID and removes it from the order.
   *
   * @param \Drupal\commerce_order\Entity\OrderItemInterface $item
   *   The order object for which to remove the order item.
   * @param int $quantity
   *   The new quantity.
   *
   * @return \Drupal\graphql_commerce\Wrapper\CommerceOrderResponse
   *   The response of the mutation operation.
   */
  private function resolveChangeQuantity(OrderItemInterface $item, int $quantity): CommerceOrderResponse {
    return $this->executeDataProducer(
      'commerce_cart_change_quantity', [
        'item' => $item,
        'quantity' => $quantity,
      ]
    );
  }

}
