<?php

declare(strict_types=1);

namespace Drupal\Tests\graphql_commerce\Kernel\DataProducer;

use Drupal\commerce_order\AvailabilityResult;
use Drupal\commerce_product\Entity\ProductVariationInterface;
use Drupal\Tests\graphql\Traits\DataProducerExecutionTrait;
use Drupal\Tests\graphql_commerce\Kernel\GraphQLCommerceOrderKernelTestBase;

/**
 * Tests the commerce_computed_availability data producer.
 *
 * @coversDefaultClass \Drupal\graphql_commerce\Plugin\GraphQL\DataProducer\ComputedAvailability
 * @group graphql_commerce
 */
class ComputedAvailabilityTest extends GraphQLCommerceOrderKernelTestBase {

  use DataProducerExecutionTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    // This makes variations with skus starting with TEST_ be unavailable.
    'commerce_order_test',
  ];

  /**
   * Tests variation availability.
   *
   * @covers ::resolve
   */
  public function testComputedAvailability(): void {
    $quantity = 1;
    $response = $this->resolveComputedAvailability($this->variation, $quantity);
    $this->assertFalse($response->isUnavailable());
    $this->assertTrue($response->isNeutral());

    // Create an unavailable product variation.
    $another_variation = $this->createEntity(
      'commerce_product_variation', [
        'type' => 'default',
        'sku' => 'TEST_unavailable',
        'price' => [
          'number' => 999,
          'currency_code' => 'USD',
        ],
        'title' => 'Some temporary title',
      ]
    );
    $response = $this->resolveComputedAvailability($another_variation, $quantity);
    $this->assertTrue($response->isUnavailable());
    $this->assertFalse($response->isNeutral());
  }

  /**
   * Execute the computed availability data producer.
   *
   * @param \Drupal\commerce_product\Entity\ProductVariationInterface $productVariation
   *   The product variation.
   * @param int|string $quantity
   *   The quantity of the product variation to add to the order.
   *
   * @return \Drupal\commerce_order\AvailabilityResult
   *   The availability result.
   */
  private function resolveComputedAvailability(ProductVariationInterface $productVariation, int|string $quantity): AvailabilityResult {
    return $this->executeDataProducer(
      'commerce_computed_availability', [
        'productVariation' => $productVariation,
        'quantity' => $quantity,
      ]
    );
  }

}
