<?php

declare(strict_types=1);

namespace Drupal\Tests\graphql_commerce\Kernel\DataProducer;

use Drupal\commerce_store\Entity\StoreInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\graphql_commerce\Wrapper\CommerceOrderResponse;
use Drupal\Tests\graphql\Traits\DataProducerExecutionTrait;
use Drupal\Tests\graphql_commerce\Kernel\GraphQLCommerceOrderKernelTestBase;

/**
 * Tests the get cart data producer.
 *
 * @coversDefaultClass \Drupal\graphql_commerce\Plugin\GraphQL\DataProducer\GetCart
 * @group graphql_commerce
 */
class GetCartTest extends GraphQLCommerceOrderKernelTestBase {

  use DataProducerExecutionTrait;

  /**
   * Tests getting the cart.
   *
   * @covers ::resolve
   */
  public function testResolveGetCart(): void {
    $response = $this->resolveGetCart(FALSE, 'xxx');
    $errors = $response->getViolations();
    $this->assertEquals(1, count($errors));
    $this->assertEquals('This order type does not exist.', $errors[0]['message']);
    $this->assertNull($response->getOrder());

    $response = $this->resolveGetCart(FALSE, 'default');
    $errors = $response->getViolations();
    $this->assertEmpty($errors);
    $this->assertNull($response->getOrder());

    $response = $this->resolveGetCart(TRUE, 'default', NULL, $this->user);
    $errors = $response->getViolations();
    $this->assertEmpty($errors);
    $this->assertEquals(1, $response->getOrder()->id());

    $another_user = $this->createUser();
    $response = $this->resolveGetCart(TRUE, 'default', NULL, $another_user);
    $errors = $response->getViolations();
    $this->assertEmpty($errors);
    $this->assertEquals(2, $response->getOrder()->id());

    $response = $this->resolveGetCart(TRUE, 'default', NULL, $this->user);
    $errors = $response->getViolations();
    $this->assertEmpty($errors);
    $this->assertEquals(1, $response->getOrder()->id());

    $another_store = $this->createStore('Another store', 'another@store.com', 'online', FALSE);
    $response = $this->resolveGetCart(FALSE, 'default', $another_store, $this->user);
    $errors = $response->getViolations();
    $this->assertEmpty($errors);
    $this->assertNull($response->getOrder());

    $response = $this->resolveGetCart(TRUE, 'default', $another_store, $this->user);
    $errors = $response->getViolations();
    $this->assertEmpty($errors);
    $this->assertEquals(3, $response->getOrder()->id());
  }

  /**
   * Execute the get cart data producer.
   *
   * @param bool $create
   *   Whether a new cart should be created if none exists.
   * @param string $orderType
   *   The order bundle.
   * @param \Drupal\commerce_store\Entity\StoreInterface|null $store
   *   The store to create the cart for.
   * @param \Drupal\Core\Session\AccountInterface|null $account
   *   The user to create the cart for.
   *
   * @return \Drupal\graphql_commerce\Wrapper\CommerceOrderResponse
   *   The order response.
   */
  private function resolveGetCart(bool $create, string $orderType, ?StoreInterface $store = NULL, ?AccountInterface $account = NULL): CommerceOrderResponse {
    return $this->executeDataProducer(
      'commerce_get_cart', [
        'create' => $create,
        'orderType' => $orderType,
        'store' => $store,
        'account' => $account,
      ]
    );
  }

}
