<?php

declare(strict_types=1);

namespace Drupal\Tests\graphql_commerce\Kernel\DataProducer;

use Drupal\Tests\graphql\Traits\DataProducerExecutionTrait;
use Drupal\Tests\graphql_commerce\Kernel\GraphQLCommerceOrderKernelTestBase;

/**
 * Tests the get carts data producer.
 *
 * @coversDefaultClass \Drupal\graphql_commerce\Plugin\GraphQL\DataProducer\GetCarts
 * @group graphql_commerce
 */
class GetCartsTest extends GraphQLCommerceOrderKernelTestBase {

  use DataProducerExecutionTrait;

  /**
   * Tests getting carts.
   *
   * @covers ::resolve
   */
  public function testResolveGetCarts(): void {
    $response = $this->resolveGetCarts('xxx');
    $this->assertEmpty($response);

    $order_type = 'default';
    $cart = $this->cartProvider->createCart($order_type, $this->store, $this->user);
    $this->cartManager->addEntity($cart, $this->variation);

    $another_store = $this->createStore('Another store', 'another@store.com', 'online', FALSE);
    $another_cart = $this->cartProvider->createCart($order_type, $another_store, $this->user);
    $this->cartManager->addEntity($another_cart, $this->variation);

    $another_user = $this->createUser();
    $third_cart = $this->cartProvider->createCart($order_type, $another_store, $another_user);
    $this->cartManager->addEntity($third_cart, $this->variation);

    $fourth_cart = $this->cartProvider->createCart($order_type, $this->store, $another_user);
    $this->cartManager->addEntity($fourth_cart, $this->variation);

    $response = $this->resolveGetCarts($order_type);
    $this->assertEquals(4, count($response));

    $response = $this->resolveGetCarts($order_type, NULL, $this->user->id());
    $this->assertEquals(2, count($response));

    $response = $this->resolveGetCarts($order_type, NULL, $another_user->id());
    $this->assertEquals(2, count($response));

    $response = $this->resolveGetCarts($order_type, $this->store->id());
    $this->assertEquals(2, count($response));

    $response = $this->resolveGetCarts($order_type, $another_store->id());
    $this->assertEquals(2, count($response));

    $response = $this->resolveGetCarts($order_type, $this->store->id(), $this->user->id());
    $this->assertEquals(1, count($response));

    $response = $this->resolveGetCarts($order_type, $this->store->id(), $another_user->id());
    $this->assertEquals(1, count($response));

    $response = $this->resolveGetCarts($order_type, $another_store->id(), $another_user->id());
    $this->assertEquals(1, count($response));

    $response = $this->resolveGetCarts($order_type, $another_store->id(), $this->user->id());
    $this->assertEquals(1, count($response));
  }

  /**
   * Execute the get carts data producer.
   *
   * @param string $orderType
   *   The order bundle.
   * @param int|string|null $storeId
   *   The store id to get the carts for.
   * @param int|string|null $accountId
   *   The user id to get the carts for.
   *
   * @return \Drupal\commerce_order\Entity\OrderInterface[]
   *   An array of orders.
   */
  private function resolveGetCarts(?string $orderType, int|string|null $storeId = NULL, int|string|null $accountId = NULL): array {
    return $this->executeDataProducer(
      'commerce_get_carts', [
        'orderType' => $orderType,
        'storeId' => $storeId,
        'accountId' => $accountId,
      ]
    );
  }

}
