<?php

declare(strict_types=1);

namespace Drupal\Tests\graphql_commerce\Kernel;

use Drupal\commerce_order\Comparator\AdjustmentComparator;
use Drupal\commerce_order\Entity\OrderItemType;
use Drupal\commerce_product\Entity\ProductVariationInterface;
use Drupal\Component\Render\FormattableMarkup;
use Drupal\user\UserInterface;
use SebastianBergmann\Comparator\Factory as PhpUnitComparatorFactory;
use Drupal\commerce_product\Entity\ProductInterface;

/**
 * Base class for commerce order related graphql tests.
 *
 * Mostly taken from \Drupal\Tests\commerce_cart\Kernel\CartKernelTestBase.
 */
abstract class GraphQLCommerceOrderKernelTestBase extends GraphQLCommerceKernelTestBase {

  /**
   * The cart manager.
   *
   * @var \Drupal\commerce_cart\CartManager
   */
  protected $cartManager;

  /**
   * The cart provider.
   *
   * @var \Drupal\commerce_cart\CartProvider
   */
  protected $cartProvider;

  /**
   * The variation to test against.
   *
   * @var \Drupal\commerce_product\Entity\ProductVariationInterface
   */
  protected ProductVariationInterface $variation;

  /**
   * A test product.
   *
   * @var \Drupal\commerce_product\Entity\ProductInterface
   */
  protected ProductInterface $product;

  /**
   * A sample user.
   *
   * @var \Drupal\user\UserInterface
   */
  protected UserInterface $user;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'entity_reference_revisions',
    'profile',
    'state_machine',
    'commerce_number_pattern',
    'commerce_product',
    'commerce_order',
    'commerce_cart',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    PhpUnitComparatorFactory::getInstance()->register(new AdjustmentComparator());

    $this->installEntitySchema('profile');
    $this->installEntitySchema('commerce_order');
    $this->installEntitySchema('commerce_order_item');
    $this->installEntitySchema('commerce_product');
    $this->installEntitySchema('commerce_product_variation');
    $this->installConfig(['commerce_product', 'commerce_order']);
    $this->installSchema('commerce_number_pattern', ['commerce_number_pattern_sequence']);

    // An order item type that doesn't need a purchasable entity.
    OrderItemType::create(
      [
        'id' => 'test',
        'label' => 'Test',
        'orderType' => 'default',
      ]
    )->save();

    $this->installConfig(['commerce_cart']);

    $this->cartManager = $this->container->get('commerce_cart.cart_manager');
    $this->cartProvider = $this->container->get('commerce_cart.cart_provider');

    // Create a product variation.
    $this->variation = $this->createEntity(
      'commerce_product_variation', [
        'type' => 'default',
        'sku' => $this->randomMachineName(),
        'price' => [
          'number' => 999,
          'currency_code' => 'USD',
        ],
        'title' => 'Some temporary title',
      ]
    );

    // We need a product too otherwise tests complain about the missing
    // backreference.
    $this->product = $this->createEntity('commerce_product', [
      'type' => 'default',
      'title' => $this->randomMachineName(),
      'stores' => [$this->store],
      'variations' => [$this->variation],
    ]);
    $this->reloadEntity($this->variation);

    // Create a user to use for orders.
    $this->user = $this->createUser();
  }

  /**
   * Creates a new entity.
   *
   * @param string $entity_type
   *   The entity type to be created.
   * @param array $values
   *   An array of settings.
   *   Example: 'id' => 'foo'.
   *
   * @return \Drupal\Core\Entity\EntityInterface
   *   A new entity.
   */
  protected function createEntity($entity_type, array $values) {
    /** @var \Drupal\Core\Entity\EntityStorageInterface $storage */
    $storage = $this->container->get('entity_type.manager')->getStorage($entity_type);
    $entity = $storage->create($values);
    $status = $entity->save();
    $this->assertEquals(
      SAVED_NEW, $status, (string) new FormattableMarkup(
        'Created %label entity %type.', [
          '%label' => $entity->getEntityType()->getLabel(),
          '%type' => $entity->id(),
        ]
      )
    );
    // The newly saved entity isn't identical to a loaded one, and would fail
    // comparisons.
    $entity = $storage->load($entity->id());

    return $entity;
  }

}
