<?php

declare(strict_types=1);

namespace Drupal\Tests\graphql_commerce\Kernel;

use Drupal\commerce_product\Entity\ProductAttribute;
use Drupal\commerce_product\Entity\ProductAttributeValue;
use Drupal\commerce_product\Entity\ProductVariationType;
use Drupal\commerce_product\Entity\ProductVariationTypeInterface;
use Drupal\field\Entity\FieldConfig;
use Drupal\Tests\graphql\Traits\DataProducerExecutionTrait;

/**
 * Base class for product attributes tests..
 */
class ProductAttributesTestBase extends GraphQLCommerceOrderKernelTestBase {

  use DataProducerExecutionTrait;

  /**
   * The color attributes values.
   *
   * @var \Drupal\commerce_product\Entity\ProductAttributeValue[]
   */
  protected $colorAttributes;

  /**
   * The attribute field manager.
   *
   * @var \Drupal\commerce_product\ProductAttributeFieldManagerInterface
   */
  protected $attributeFieldManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installEntitySchema('commerce_product_attribute');
    $this->installEntitySchema('commerce_product_attribute_value');
    $this->attributeFieldManager = $this->container->get('commerce_product.attribute_field_manager');

    $variation_type = ProductVariationType::load('default');
    $this->colorAttributes = $this->createAttributeSet($variation_type, 'color', [
      'black' => 'Black',
      'blue' => 'Blue',
      'green' => 'Green',
      'red' => 'Red',
      'white' => 'White',
      'yellow' => 'Yellow',
    ]);
  }

  /**
   * Creates an attribute field and set of attribute values.
   *
   * Copy-pasted from ProductVariationAttributeMapperTest.
   *
   * @param \Drupal\commerce_product\Entity\ProductVariationTypeInterface $variation_type
   *   The variation type.
   * @param string $name
   *   The attribute field name.
   * @param array $options
   *   Associative array of key name values. [red => Red].
   * @param bool $required
   *   Whether the created attribute should be required.
   *
   * @return \Drupal\commerce_product\Entity\ProductAttributeValueInterface[]
   *   Array of attribute entities.
   */
  protected function createAttributeSet(ProductVariationTypeInterface $variation_type, $name, array $options, $required = TRUE) {
    $attribute = ProductAttribute::create([
      'id' => $name,
      'label' => ucfirst($name),
    ]);
    $attribute->save();
    $this->attributeFieldManager->createField($attribute, $variation_type->id());
    // The field is always created as required by default.
    if (!$required) {
      $field = FieldConfig::loadByName('commerce_product_variation', $variation_type->id(), 'attribute_' . $name);
      $field->setRequired(FALSE);
      $field->save();
    }

    $attribute_set = [];
    foreach ($options as $key => $value) {
      $attribute_set[$key] = $this->createAttributeValue($name, $value);
    }

    return $attribute_set;
  }

  /**
   * Creates an attribute value.
   *
   * Copy-pasted from ProductVariationAttributeMapperTest.
   *
   * @param string $attribute
   *   The attribute ID.
   * @param string $name
   *   The attribute value name.
   *
   * @return \Drupal\commerce_product\Entity\ProductAttributeValueInterface
   *   The attribute value entity.
   */
  protected function createAttributeValue($attribute, $name) {
    $attribute_value = ProductAttributeValue::create([
      'attribute' => $attribute,
      'name' => $name,
    ]);
    $attribute_value->save();

    return $attribute_value;
  }

}
