<?php

declare(strict_types=1);

namespace Drupal\Tests\graphql_commerce\Kernel\SchemaExtension;

use Drupal\graphql\Entity\ServerInterface;
use Drupal\Tests\graphql_commerce\Kernel\GraphQLCommerceOrderKernelTestBase;
use GraphQL\Server\OperationParams;

/**
 * Tests the commerce_availability extension.
 *
 * @group graphql_core_schema
 */
class ComputedAvailabilityExtensionTest extends GraphQLCommerceOrderKernelTestBase {

  /**
   * The GraphQL server.
   */
  protected ServerInterface $server;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    // This makes variations with skus starting with TEST_ be unavailable.
    'commerce_order_test',
  ];

  /**
   * Test the product availability.
   */
  public function testComputedAvailabilityExtension(): void {
    $this->server = $this
      ->getCoreComposableServerBuilder()
      ->enableValueFields()
      ->enableExtension('commerce_availability')
      ->enableExtension('entity_query')
      ->enableEntityType('commerce_product_variation', ['variation_id'], ['default'])
      ->enableEntityType('user', ['uid'])
      ->createServer();

    // entityById does not have a "skip access checks" switch.
    $user = $this->setUpCurrentUser();
    $role = $this->createRole(['view commerce_product']);
    $user->addRole($role);
    $this->setCurrentUser($user);

    // Create an unavailable product variation.
    $another_variation = $this->createEntity(
      'commerce_product_variation', [
        'type' => 'default',
        'sku' => 'TEST_unavailable',
        'price' => [
          'number' => 999,
          'currency_code' => 'USD',
        ],
        'title' => 'Some temporary title',
        'product_id' => $this->product->id(),
      ]
    );

    // Test an available variation.
    $quantity = 1;
    $variables = [
      'quantity' => $quantity,
      'productVariation' => $this->variation->id(),
    ];
    $expected = [
      'id' => $this->variation->id(),
      'computedAvailability' => [
        'isNeutral' => TRUE,
        'isUnavailable' => FALSE,
      ],
    ];
    $this->assertComputedAvailability($variables, $expected);

    // Unavailable variation.
    $variables['productVariation'] = $another_variation->id();
    $expected = [
      'id' => $another_variation->id(),
      'computedAvailability' => [
        'isNeutral' => FALSE,
        'isUnavailable' => TRUE,
      ],
    ];
    $this->assertComputedAvailability($variables, $expected);
  }

  /**
   * Assert the computed availability.
   *
   * @param array $variables
   *   The variables to pass to the query.
   * @param array $expected
   *   The expected values.
   */
  private function assertComputedAvailability(array $variables, array $expected) {
    $query = <<<GQL
    query computedAvailability(\$productVariation: ID!, \$quantity: Int!) {
      entityById(entityType: COMMERCE_PRODUCT_VARIATION, id: \$productVariation) {
        ... on CommerceProductVariationDefault {
          id
          computedAvailability(quantity: \$quantity) {
            isNeutral
            isUnavailable
          }
        }
      }
    }
    GQL;

    $params = OperationParams::create(
      [
        'query' => $query,
        'variables' => $variables,
      ]
    );
    $result = $this->server->executeOperation($params);
    $data = $result->data['entityById'];
    $this->assertEquals($data, $expected);
  }

}
