<?php

declare(strict_types=1);

namespace Drupal\Tests\graphql_commerce\Kernel\SchemaExtension;

use Drupal\graphql\Entity\ServerInterface;
use Drupal\Tests\graphql_commerce\Kernel\GraphQLCommerceOrderKernelTestBase;
use GraphQL\Server\OperationParams;

/**
 * Tests the commerce_get_cart extension.
 *
 * @group graphql_core_schema
 */
class GetCartExtensionTest extends GraphQLCommerceOrderKernelTestBase {

  /**
   * The GraphQL server.
   */
  protected ServerInterface $server;

  /**
   * Test the Url type resolvers.
   */
  public function testGetCartResolver(): void {
    $this->server = $this
      ->getCoreComposableServerBuilder()
      ->enableValueFields()
      ->enableExtension('commerce')
      ->enableEntityType('commerce_order', ['id'], ['default'])
      ->enableEntityType('commerce_product_variation')
      ->enableEntityType('commerce_product')
      ->enableEntityType('commerce_store', ['id'], ['default'])
      ->enableEntityType('user', ['uid'])
      ->createServer();

    // Try a nonexistent order type.
    $variables = [
      'create' => FALSE,
      'orderType' => 'xxx',
      'account' => $this->user->id(),
      'store' => $this->store->id(),
    ];
    $expected = [
      'order' => '',
      'errors' => [
        [
          'message' => 'This order type does not exist.',
        ],
      ],
    ];
    $this->assertGetCart($variables, $expected);

    // Existing order type, do not create cart.
    $variables['orderType'] = 'default';
    $expected['errors'] = [];
    $this->assertGetCart($variables, $expected);

    // Create cart.
    $variables['create'] = TRUE;
    $expected['order'] = ['id' => 1];
    $this->assertGetCart($variables, $expected);

    // Different users should get a different cart.
    $another_user = $this->createUser();
    $variables['account'] = $another_user->id();
    $expected['order']['id'] = 2;
    $this->assertGetCart($variables, $expected);

    // Original user should get the original cart.
    $variables['account'] = $this->user->id();
    $expected['order']['id'] = 1;
    $this->assertGetCart($variables, $expected);

    // For a new store we don't yet have a cart.
    $another_store = $this->createStore('Another store', 'another@store.com', 'online', FALSE);
    $variables['create'] = FALSE;
    $variables['store'] = $another_store->id();
    $expected['order'] = '';
    $this->assertGetCart($variables, $expected);

    // ...unless we create it.
    $variables['create'] = TRUE;
    $expected['order'] = ['id' => 3];
    $this->assertGetCart($variables, $expected);

    // Not passing in a store should use the default store.
    $variables['store'] = '';
    $expected['order']['id'] = 1;
    $this->assertGetCart($variables, $expected);
  }

  /**
   * Assert the resolved get cart type.
   *
   * @param array $variables
   *   The variables to pass to the query.
   * @param array $expected
   *   The expected values.
   */
  private function assertGetCart(array $variables, array $expected) {
    $query = <<<GQL
    query getCart(\$create: Boolean!, \$orderType: String!, \$account: ID, \$store: ID) {
      commerceGetCart(
        create: \$create, 
        orderType: \$orderType,  
        account: \$account, 
        store: \$store) {
        errors
        order {
          id
        }
      }
    }
    GQL;

    $params = OperationParams::create(
      [
        'query' => $query,
        'variables' => $variables,
      ]
    );
    $result = $this->server->executeOperation($params);
    $data = $result->data['commerceGetCart'];
    $this->assertEquals($data, $expected);
  }

}
