<?php

declare(strict_types=1);

namespace Drupal\graphql_compose_layout_builder\Plugin\GraphQL\DataProducer;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\graphql\Attribute\DataProducer;
use Drupal\graphql\GraphQL\Execution\FieldContext;
use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;
use Drupal\graphql_compose_layout_builder\Wrapper\LayoutBuilderSection;
use Drupal\layout_builder\SectionStorage\SectionStorageManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Layout Builder section loader.
 */
#[DataProducer(
  id: "layout_builder_sections",
  name: new TranslatableMarkup("Layout Builder sections"),
  description: new TranslatableMarkup("Get layout builder sections."),
  produces: new ContextDefinition(
    data_type: "any",
    label: new TranslatableMarkup("Section"),
  ),
  consumes: [
    "contexts" => new ContextDefinition(
      data_type: "any",
      label: new TranslatableMarkup("Layout builder contexts"),
    ),
  ],
)]
class LayoutBuilderSections extends DataProducerPluginBase implements ContainerFactoryPluginInterface {

  /**
   * The layout builder section storage manager.
   *
   * @var \Drupal\layout_builder\SectionStorage\SectionStorageManagerInterface
   */
  protected SectionStorageManagerInterface $sectionStorageManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition
    );

    $instance->sectionStorageManager = $container->get('plugin.manager.layout_builder.section_storage');

    return $instance;
  }

  /**
   * Return sections for a layout builder context.
   *
   * @param array|null $contexts
   *   The contexts to use to load the sections.
   * @param \Drupal\graphql\GraphQL\Execution\FieldContext $context
   *   Cacheability context for this request.
   *
   * @return \Drupal\graphql_compose_layout_builder\Wrapper\LayoutBuilderSection[]|null
   *   Sections loaded by context.
   */
  public function resolve(?array $contexts, FieldContext $context) {

    if (is_null($contexts)) {
      return NULL;
    }

    $section_storage = $this->sectionStorageManager->findByContext($contexts, $context);
    if (!$section_storage) {
      return NULL;
    }

    $context->addCacheableDependency($section_storage);

    $sections = [];
    foreach ($section_storage->getSections() as $delta => $section) {
      $sections[] = new LayoutBuilderSection($section, $section_storage, $delta);
    }

    return $sections;
  }

}
