<?php

declare(strict_types=1);

namespace Drupal\graphql_compose;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\TranslatableInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\graphql\GraphQL\Execution\FieldContext;

/**
 * A wrapper for entity repository language translation trait.
 */
trait EntityTranslateTrait {

  /**
   * The language manager service.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * Fetch the language manager service.
   *
   * @return \Drupal\Core\Language\LanguageManagerInterface
   *   The language manager service.
   */
  protected function getLanguageManager(): LanguageManagerInterface {
    return $this->languageManager ??= \Drupal::service('language_manager');
  }

  /**
   * Fetch the module handler service.
   *
   * @return \Drupal\Core\Extension\ModuleHandlerInterface
   *   The module handler service.
   */
  protected function getModuleHandler(): ModuleHandlerInterface {
    return $this->moduleHandler ??= \Drupal::service('module_handler');
  }

  /**
   * Translate an entity to the requested language.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity to translate.
   * @param string $langcode
   *   (optional) The language code to translate to.
   *
   * @return \Drupal\Core\Entity\EntityInterface|null
   *   The translated entity, or NULL if the translation is missing,
   */
  protected function getTranslation(EntityInterface $entity, ?string $langcode = NULL): ?EntityInterface {

    // Site is not multilingual, return the entity provided.
    if (!$this->getLanguageManager()->isMultilingual()) {
      return $entity;
    }

    if ($entity instanceof TranslatableInterface && $langcode && $langcode !== $entity->language()->getId()) {
      $variables = [
        'entity' => $entity,
        'langcode' => $langcode,
      ];

      if ($entity->hasTranslation($langcode)) {
        $entity = $entity->getTranslation($langcode);
        $entity->addCacheContexts(["static:language:$langcode"]);
      }

      // Allow modules to alter the entity before returning it.
      // @see hook_graphql_compose_entity_translate_alter()
      $this->getModuleHandler()->alter('graphql_compose_entity_translate', $entity, $variables);
    }

    return $entity;
  }

  /**
   * Get the current language for a query.
   *
   * @param \Drupal\graphql\GraphQL\Execution\FieldContext $context
   *   The field context.
   * @param string|null $preferred
   *   The preferred language code.
   * @param string|null $fallback
   *   The Drupal\Core\Language\LanguageInterface type to use.
   *   ie LanguageInterface::TYPE_INTERFACE.
   *
   * @return string|null
   *   The current language code.
   */
  protected function getCurrentLanguage(FieldContext $context, ?string $preferred = NULL, ?string $fallback = NULL): ?string {
    $variables = [
      'context' => $context,
      'preferred' => $preferred,
      'fallback' => $fallback,
      'fallback_used' => FALSE,
    ];

    // Fallback from preferred to context language.
    $langcode = $preferred ?: $context->getContextLanguage();

    // Sometimes (ie redirects, revisions) we need a value.
    if (!$langcode && $fallback) {
      $langcode = $this->getLanguageManager()->getCurrentLanguage($fallback)->getId();
      $variables['fallback_used'] = TRUE;
    }

    // Allow modules to alter the entity before returning it.
    // @see hook_graphql_compose_current_language_alter()
    $this->getModuleHandler()->alter('graphql_compose_current_language', $langcode, $variables);

    return $langcode;
  }

}
