<?php

declare(strict_types=1);

namespace Drupal\graphql_compose\Plugin\GraphQL\SchemaExtension;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\graphql\Attribute\SchemaExtension;
use Drupal\graphql\GraphQL\ResolverBuilder;
use Drupal\graphql\GraphQL\ResolverRegistryInterface;
use Drupal\graphql_compose\Utility\ComposeConfig;

/**
 * Languages are a common occurrence. Map LanguageInterface objects in schema.
 *
 * @internal
 */
#[SchemaExtension(
  id: "graphql_compose_language",
  name: new TranslatableMarkup("GraphQL Compose Languages"),
  description: new TranslatableMarkup("Add language support to schema."),
  schema: "graphql_compose",
  priority: 75,
)]
class LanguageSchemaExtension extends ResolverOnlySchemaExtensionPluginBase implements ContainerFactoryPluginInterface {

  /**
   * {@inheritdoc}
   *
   * We need to override the resolvers for LanguageInterface
   * because the Language is not a real entity.
   *
   * We combine ConfigurableLanguage and Language into one type, to give the
   * schema some consistency.
   *
   * @see Drupal\graphql_compose\Plugin\GraphQLCompose\EntityType\ConfigurableLanguage
   */
  public function registerResolvers(ResolverRegistryInterface $registry): void {
    $builder = new ResolverBuilder();

    // If the ConfigurableLanguage entity type is not available,
    // we need to register a LanguageInterface type resolver.
    if (!$this->moduleHandler->moduleExists('language')) {
      $registry->addTypeResolver(
        'Language',
        fn($language) => $language instanceof LanguageInterface ? $language : NULL,
      );
    }

    $registry->addFieldResolver(
      'Language',
      'id',
      $builder->callback(fn (LanguageInterface $language) => $language->getId())
    );

    $expose_entity_ids = ComposeConfig::get('settings.expose_entity_ids', FALSE);

    if ($expose_entity_ids) {
      // It's a duplication of the id, but some would expect this to be present.
      $registry->addFieldResolver(
        'Language',
        'uuid',
        $builder->callback(fn (LanguageInterface $language) => $language->getId())
      );
    }

    $registry->addFieldResolver(
      'Language',
      'name',
      $builder->callback(fn (LanguageInterface $language) => $language->getName())
    );

    $registry->addFieldResolver(
      'Language',
      'direction',
      $builder->callback(fn (LanguageInterface $language) => $language->getDirection())
    );

    $registry->addFieldResolver(
      'Language',
      'weight',
      $builder->callback(fn (LanguageInterface $language) => $language->getWeight())
    );

    if ($this->languageManager->isMultilingual()) {
      $registry->addFieldResolver(
        'SchemaInformation',
        'languages',
        $builder->callback(fn () => $this->languageManager->getLanguages())
      );

      foreach ($this->gqlEntityTypeManager->getPluginInstances() as $entity_type_plugin) {
        foreach ($entity_type_plugin->getBundles() as $bundle) {

          if (!$bundle->isTranslatableContent()) {
            continue;
          }

          $registry->addFieldResolver(
            $bundle->getTypeSdl(),
            'translations',
            $builder->compose(
              $builder->produce('entity_translations')
                ->map('entity', $builder->fromParent()),
              $builder->callback(
                fn (?array $translations) => array_filter($translations ?: [])
              )
            )
          );
        }
      }

      $registry->addFieldResolver(
        'Translation',
        'title',
        $builder->produce('entity_label')
          ->map('entity', $builder->fromParent())
      );

      $registry->addFieldResolver(
        'Translation',
        'langcode',
        $builder->produce('entity_language')
          ->map('entity', $builder->fromParent())
      );

      $registry->addFieldResolver(
        'Translation',
        'path',
        $builder->cond([
          [
            $builder->callback(fn (EntityInterface $entity) => $entity->isNew()),
            $builder->fromValue(NULL),
          ], [
            $builder->fromValue(TRUE),
            $builder->compose(
              $builder->produce('entity_url')
                ->map('entity', $builder->fromParent()),

              $builder->produce('url_path')
                ->map('url', $builder->fromParent()),
            ),
          ],
        ]),
      );
    }

  }

}
