<?php

declare(strict_types=1);

namespace Drupal\graphql_compose\Plugin\GraphQLCompose\SchemaType;

use Drupal\graphql_compose\Attribute\SchemaType;
use Drupal\graphql_compose\Plugin\GraphQLCompose\GraphQLComposeSchemaTypeBase;
use GraphQL\Type\Definition\ObjectType;
use GraphQL\Type\Definition\Type;

/**
 * {@inheritdoc}
 */
#[SchemaType(
  id: "SmartDate",
)]
class SmartDateType extends GraphQLComposeSchemaTypeBase {

  /**
   * {@inheritdoc}
   */
  public function getTypes(): array {
    $types = [];

    if (!$this->moduleHandler->moduleExists('smart_date')) {
      return $types;
    }

    // The smart_date module has a dependency on datetime_range,
    // but doesn't require it (for unknown reasons).
    // If the datetime_range module is enabled, we will provide recurrence
    // by re-using the Schema DateRange type.
    $hasRecurrence = $this->moduleHandler->moduleExists('smart_date_recur');
    $hasDatetimeRange = $this->moduleHandler->moduleExists('datetime_range');

    $types[] = new ObjectType([
      'name' => $this->getPluginId(),
      'description' => (string) $this->t('A Date range has a start and an end.'),
      'fields' => fn() => array_filter([
        'start' => [
          'type' => static::type('DateTime'),
          'description' => (string) $this->t('The start of the date range.'),
        ],
        'end' => [
          'type' => static::type('DateTime'),
          'description' => (string) $this->t('The end of the date range.'),
        ],
        'duration' => [
          'type' => Type::float(),
          'description' => (string) $this->t('The duration, in minutes.'),
        ],
        'timezone' => [
          'type' => Type::string(),
          'description' => (string) $this->t('The timezone of the date range.'),
        ],
        'recurrence' => $hasRecurrence && $hasDatetimeRange
          ? [
            'type' => Type::nonNull(Type::listOf(Type::nonNull(static::type('DateRange')))),
            'description' => (string) $this->t('The recurrence for the date range.'),
          ] : NULL,
      ]),
    ]);

    return $types;
  }

}
