<?php

declare(strict_types=1);

namespace Drupal\graphql_compose\Plugin;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\DefaultPluginManager;
use Drupal\graphql_compose\Annotation\GraphQLComposeEntityType;
use Drupal\graphql_compose\Attribute\EntityType;
use Drupal\graphql_compose\Utility\ComposeContext;
use Drupal\graphql_compose\Utility\ComposeProviders;
use Drupal\graphql_compose\Plugin\GraphQLCompose\GraphQLComposeEntityTypeInterface;

/**
 * Manager that collects and exposes GraphQL Compose entity plugins.
 *
 * A entity type is a plugin that defines how to resolve a Drupal Entity Type.
 */
class GraphQLComposeEntityTypeManager extends DefaultPluginManager {

  /**
   * Static storage of instances.
   *
   * @var \Drupal\graphql_compose\Plugin\GraphQLCompose\GraphQLComposeEntityTypeInterface[]
   */
  private array $instances = [];

  /**
   * Constructs a GraphQLComposeEntityTypeManager object.
   *
   * @param \Traversable $namespaces
   *   An object that implements \Traversable which contains the root paths
   *   keyed by the corresponding namespace to look for plugin implementations.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_backend
   *   The cache backend.
   * @param array $config
   *   The configuration service parameter.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(
    \Traversable $namespaces,
    ModuleHandlerInterface $module_handler,
    CacheBackendInterface $cache_backend,
    array $config,
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct(
      'Plugin/GraphQLCompose/EntityType',
      $namespaces,
      $module_handler,
      GraphQLComposeEntityTypeInterface::class,
      EntityType::class,
      GraphQLComposeEntityType::class
    );

    $this->alterInfo('graphql_compose_entity_type');
    $this->useCaches(empty($config['development']));
    $this->setCacheBackend($cache_backend, 'graphql_compose_entity_type', ['graphql_compose_entity_type']);
  }

  /**
   * Get a plugin instance.
   *
   * @param string $plugin_id
   *   Type to get settings for. Eg 'media'.
   *
   * @return \Drupal\graphql_compose\Plugin\GraphQLCompose\GraphQLComposeEntityTypeInterface|null
   *   The plugin instance.
   */
  public function getPluginInstance(string $plugin_id): ?GraphQLComposeEntityTypeInterface {

    // Assume the plugin ID is the same as the entity type ID.
    $entity_type_id = $plugin_id;

    // Limit the plugin to the providers that are enabled.
    if (!$this->hasDefinition($plugin_id) || !$this->entityTypeManager->hasDefinition($plugin_id)) {
      // Deal with derivatives, look at the id.
      $definitions = $this->getDefinitions();
      foreach ($definitions as $derivative_id => $definition) {
        if ($definition['id'] === $plugin_id || $derivative_id === $plugin_id) {
          $entity_type_id = $definition['id'];
          $plugin_id = $derivative_id;
          break;
        }
      }
    }

    if ($this->hasDefinition($plugin_id) && $this->entityTypeManager->hasDefinition($entity_type_id)) {
      $definition = $this->getDefinition($plugin_id, TRUE);
      if (ComposeProviders::has($definition['provider'])) {
        return $this->createInstance($plugin_id);
      }
    }

    return NULL;
  }

  /**
   * Get all plugin instances.
   *
   * @return \Drupal\graphql_compose\Plugin\GraphQLCompose\GraphQLComposeEntityTypeInterface[]
   *   All plugin instances.
   */
  public function getPluginInstances(): array {
    $server_id = ComposeContext::getServerId();

    array_map(
      $this->getPluginInstance(...),
      array_keys($this->getDefinitions())
    );

    return $this->instances[$server_id] ?? [];
  }

  /**
   * Hijack the createInstance to register the types and extensions.
   *
   * @param string $plugin_id
   *   The plugin id.
   * @param array $configuration
   *   The plugin configuration.
   *
   * @return \Drupal\graphql_compose\Plugin\GraphQLCompose\GraphQLComposeEntityTypeInterface
   *   The plugin instance.
   */
  public function createInstance($plugin_id, array $configuration = []) {
    $server_id = ComposeContext::getServerId();

    if (!isset($this->instances[$server_id][$plugin_id])) {
      $plugin = parent::createInstance($plugin_id, $configuration);
      $this->instances[$server_id][$plugin_id] = $plugin;
      $plugin->registerTypes();
    }

    return $this->instances[$server_id][$plugin_id];
  }

}
