<?php

declare(strict_types=1);

namespace Drupal\graphql_compose\Utility;

use Drupal\graphql\Entity\Server;

/**
 * Utility for GraphQL Compose providers.
 *
 * Use during GraphQL runtime to get information on current server context.
 */
class ComposeContext {


  /**
   * The server id this context is registered for.
   *
   * @var string|null
   */
  private static ?string $serverId = NULL;

  /**
   * Sets the server id for this context, and resets the config.
   *
   * @param string|null $server_id
   *   The server id to set.
   */
  public static function setServerId(?string $server_id): void {
    self::$serverId = $server_id;
  }

  /**
   * Gets the server id for this context.
   *
   * @return string|null
   *   The server id.
   */
  public static function getServerId(): ?string {
    if (!self::$serverId) {
      // Lets try get from current route.
      $currentRoute = \Drupal::service('current_route_match');
      self::setServerId($currentRoute->getParameter('graphql_server')?->id());
    }
    return self::$serverId;
  }

  /**
   * Gets the schema configuration for this context.
   *
   * @return array
   *   The schema configuration.
   */
  public static function getServerSchemaConfiguration(): array {
    static $config;
    $id = self::getServerId();

    if ($id && !isset($config[$id])) {
      $server = Server::load($id);
      if ($server) {
        $schema_configuration = $server->get('schema_configuration');
        $config[$id] = $schema_configuration['graphql_compose'] ?? [];
      }
    }

    return $config[$id] ?? [];
  }

}
