<?php

declare(strict_types=1);

namespace Drupal\Tests\graphql_compose\Functional\Contrib;

use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\Tests\graphql_compose\Functional\GraphQLComposeBrowserTestBase;
use Drupal\node\NodeInterface;

/**
 * Test smart date module integration with GraphQL Compose.
 *
 * @group graphql_compose
 */
class SmartDateTest extends GraphQLComposeBrowserTestBase {

  /**
   * The test node.
   *
   * @var \Drupal\node\NodeInterface
   */
  protected NodeInterface $node;

  /**
   * The test date.
   *
   * @var int
   */
  protected int $now;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'datetime',
    'datetime_range',
    'smart_date',
    'smart_date_recur',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->createContentType([
      'type' => 'test',
      'name' => 'Test node type',
    ]);

    FieldStorageConfig::create([
      'field_name' => 'field_smart_date',
      'type' => 'smartdate',
      'entity_type' => 'node',
    ])->save();

    FieldConfig::create([
      'field_name' => 'field_smart_date',
      'entity_type' => 'node',
      'bundle' => 'test',
      'label' => 'Smart date',
    ])->save();

    $this->now = \Drupal::time()->getRequestTime();

    $this->node = $this->createNode([
      'type' => 'test',
      'title' => 'Test',
      'created' => $this->now,
      'changed' => $this->now,
      'field_smart_date' => [
        'value' => $this->now,
        'end_value' => $this->now + 86400,
        'timezone' => 'UTC',
        'duration' => 86400,
      ],
      'status' => 1,
    ]);

    $this->setEntityConfig('node', 'test', [
      'enabled' => TRUE,
      'query_load_enabled' => TRUE,
    ]);

    $this->setFieldConfig('node', 'test', 'field_smart_date', [
      'enabled' => TRUE,
    ]);
  }

  /**
   * Test load entity by id.
   */
  public function testSmartDateField(): void {
    $query = <<<GQL
      query {
        node(id: "{$this->node->uuid()}") {
          ... on NodeTest {
            smartDate {
              start {
                timestamp
              }
              end {
                timestamp
              }
              timezone
              duration
            }
          }
        }
      }
    GQL;

    $content = $this->executeQuery($query);

    $node = $content['data']['node'];

    $this->assertEquals($this->now, $node['smartDate']['start']['timestamp']);
    $this->assertEquals($this->now + 86400, $node['smartDate']['end']['timestamp']);
    $this->assertEquals('UTC', $node['smartDate']['timezone']);
    $this->assertEquals(86400, $node['smartDate']['duration']);
  }

}
