<?php

declare(strict_types=1);

namespace Drupal\graphql_compose_blocks\Plugin\GraphQL\DataProducer;

use Drupal\Core\Block\BlockPluginInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\block_content\Plugin\Block\BlockContentBlock;
use Drupal\Component\Utility\DeprecationHelper;
use Drupal\Core\Access\AccessibleInterface;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\Plugin\Context\ContextInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\graphql\Attribute\DataProducer;
use Drupal\graphql\GraphQL\Execution\FieldContext;
use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;
use Drupal\layout_builder\Plugin\Block\InlineBlock;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Load drupal block content entities.
 */
#[DataProducer(
  id: "block_content_entity_load",
  name: new TranslatableMarkup("Block Content entity"),
  description: new TranslatableMarkup("Load a block's entity."),
  produces: new ContextDefinition(
    data_type: "any",
    label: new TranslatableMarkup("Entity")
  ),
  consumes: [
    "block_instance" => new ContextDefinition(
      data_type: "any",
      label: new TranslatableMarkup("Block instance")
    ),
  ]
)]
class BlockContentEntityLoad extends DataProducerPluginBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs a BlockEntityLoad object.
   *
   * @param array $configuration
   *   The plugin configuration.
   * @param string $plugin_id
   *   The plugin id.
   * @param mixed $plugin_definition
   *   The plugin definition.
   * @param \Drupal\Core\Entity\EntityRepositoryInterface $entityRepository
   *   Drupal entity repository.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected EntityRepositoryInterface $entityRepository,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity.repository'),
    );
  }

  /**
   * Resolve the layout definition.
   *
   * @param \Drupal\Core\Block\BlockPluginInterface $block_instance
   *   The block instance.
   * @param \Drupal\graphql\GraphQL\Execution\FieldContext $context
   *   The cache context.
   *
   * @return \Drupal\Core\Entity\EntityInterface|null
   *   The block entity.
   */
  public function resolve(BlockPluginInterface $block_instance, FieldContext $context): ?EntityInterface {

    $entity = NULL;

    if ($block_instance instanceof InlineBlock) {
      // Tap into the block plugin and use its protected function.
      $method = new \ReflectionMethod($block_instance::class, 'getEntity');
      $method->setAccessible(TRUE);

      $entity = $method->invoke($block_instance);
    }
    elseif ($block_instance instanceof BlockContentBlock) {
      $uuid = $block_instance->getDerivativeId();
      $entity = $this->entityRepository->loadEntityByUuid('block_content', $uuid);
    }
    else {
      return NULL;
    }

    if (!$entity) {
      return NULL;
    }

    $context->addCacheableDependency($entity);

    // Handle RefinableDependentAccessInterface move between versions.
    // https://www.drupal.org/node/3527501
    $refinableDependentClassName = DeprecationHelper::backwardsCompatibleCall(
      currentVersion: \Drupal::VERSION,
      deprecatedVersion: '11.2',
      currentCallable: fn() => '\Drupal\Core\Access\RefinableDependentAccessInterface',
      deprecatedCallable: fn() => '\Drupal\block_content\Access\RefinableDependentAccessInterface',
    );

    // Get the parent entity from layout_builder_contexts as access dependency.
    // https://www.drupal.org/project/graphql_compose/issues/3556558
    if (is_subclass_of($entity, $refinableDependentClassName, FALSE)) {
      // Check for and get the context for Layout builder blocks.
      if ($context->hasContextValue('layout_builder_contexts')) {
        // Get the layout builder entity from the context.
        $layout_builder_entity = $context->getContextValue('layout_builder_contexts')['entity'] ?? NULL;
        if ($layout_builder_entity instanceof ContextInterface) {
          // Replace the access dependency with the parent entity.
          $parent_entity = $layout_builder_entity->getContextValue();
          if ($parent_entity instanceof AccessibleInterface) {
            $entity->setAccessDependency($parent_entity);
          }
        }
      }
    }

    $access = $entity->access('view', NULL, TRUE);
    $context->addCacheableDependency($access);

    if (!$access->isAllowed()) {
      return NULL;
    }

    return $entity;
  }

}
