<?php

declare(strict_types=1);

namespace Drupal\graphql_compose_layout_builder\Wrapper;

use Drupal\layout_builder\Section;
use Drupal\layout_builder\SectionStorageInterface;

/**
 * A wrapped layout builder section.
 *
 * We wrap in the storage and display delta.
 */
class LayoutBuilderSection {

  /**
   * Construct a section wrapper.
   *
   * @param \Drupal\layout_builder\Section $section
   *   The section.
   * @param \Drupal\layout_builder\SectionStorageInterface $storage
   *   The section storage.
   * @param int $delta
   *   The section delta.
   */
  public function __construct(
    protected Section $section,
    protected SectionStorageInterface $storage,
    protected int $delta,
  ) {}

  /**
   * The section storage id.
   *
   * @return string
   *   The section storage id.
   */
  public function id(): string {
    $parts = [
      $this->storage->getStorageId(),
      $this->storage->getStorageType(),
      'section',
      $this->delta,
    ];
    return implode('.', $parts);
  }

  /**
   * The display position of this section.
   *
   * @return int
   *   The section delta.
   */
  public function getDelta(): int {
    return $this->delta;
  }

  /**
   * The section storage.
   *
   * @return \Drupal\layout_builder\SectionStorageInterface
   *   The section storage.
   */
  public function getStorage(): SectionStorageInterface {
    return $this->storage;
  }

  /**
   * The layout id of this section.
   *
   * @return string
   *   The layout id.
   */
  public function getLayoutId(): string {
    return $this->section->getLayoutId();
  }

  /**
   * The layout settings of this section.
   *
   * @return mixed[]
   *   The layout settings.
   */
  public function getLayoutSettings(): array {
    return $this->section->getLayoutSettings();
  }

  /**
   * The components of this section.
   *
   * @return \Drupal\layout_builder\SectionComponent[]
   *   The section components.
   */
  public function getComponents(): array {
    return $this->section->getComponents();
  }

}
