<?php

declare(strict_types=1);

namespace Drupal\graphql_compose\Plugin\GraphQL\DataProducer;

use Drupal\Component\Uuid\Uuid;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\Plugin\Context\EntityContextDefinition;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\graphql\Attribute\DataProducer;
use Drupal\graphql\Plugin\DataProducerPluginManager;
use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;
use Drupal\graphql_compose\Utility\ComposeConfig;
use GraphQL\Deferred;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Loads an entity by UUID or ID if allowed.
 */
#[DataProducer(
  id: "entity_load_by_uuid_or_id",
  name: new TranslatableMarkup("Load entity by uuid or ID"),
  description: new TranslatableMarkup("Loads a single entity by uuid or ID."),
  produces: new ContextDefinition(
    data_type: "entity",
    label: new TranslatableMarkup("Entity"),
  ),
  consumes: [
    "type" => new ContextDefinition(
      data_type: "string",
      label: new TranslatableMarkup("Entity type"),
    ),
    "identifier" => new ContextDefinition(
      data_type: "string",
      label: new TranslatableMarkup("Unique identifier"),
    ),
    "language" => new ContextDefinition(
      data_type: "string",
      label: new TranslatableMarkup("Entity language"),
      required: FALSE,
    ),
    "bundles" => new ContextDefinition(
      data_type: "string",
      label: new TranslatableMarkup("Entity bundle(s)"),
      multiple: TRUE,
      required: FALSE,
    ),
    "access" => new ContextDefinition(
      data_type: "boolean",
      label: new TranslatableMarkup("Check access"),
      required: FALSE,
      default_value: TRUE,
    ),
    "access_user" => new EntityContextDefinition(
      data_type: "entity:user",
      label: new TranslatableMarkup("User"),
      required: FALSE,
      default_value: NULL,
    ),
    "access_operation" => new ContextDefinition(
      data_type: "string",
      label: new TranslatableMarkup("Operation"),
      required: FALSE,
      default_value: "view",
    ),
  ],
)]
class EntityLoadByUuidOrId extends DataProducerPluginBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs a new EntityLoadByUuidOrId instance.
   *
   * @param array $configuration
   *   The plugin configuration.
   * @param string $plugin_id
   *   The plugin ID.
   * @param mixed $plugin_definition
   *   The plugin definition.
   * @param \Drupal\graphql\Plugin\DataProducerPluginManager $pluginManager
   *   The data producer plugin manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    $plugin_definition,
    protected DataProducerPluginManager $pluginManager,
    protected ConfigFactoryInterface $configFactory,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('plugin.manager.graphql.data_producer'),
      $container->get('config.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function resolve(...$args): ?Deferred {

    // Users select how they want to display ids.
    $expose_entity_ids = ComposeConfig::get('settings.expose_entity_ids', FALSE);

    $plugin_id = 'entity_load_by_uuid';

    if ($expose_entity_ids && !Uuid::isValid($args[1])) {
      $plugin_id = 'entity_load';
    }

    return $this->pluginManager
      ->createInstance($plugin_id, $this->configuration)
      ->resolve(...$args);
  }

}
