<?php

declare(strict_types=1);

namespace Drupal\graphql_compose\Utility;

use Drupal\Core\Config\ImmutableConfig;

/**
 * Utility for GraphQL Compose providers.
 *
 * Use during GraphQL runtime to get configuration for the
 * current server context.
 */
class ComposeConfig {

  /**
   * Gets the server configuration name for this context.
   *
   * @return string|null
   *   The server configuration name, or NULL if not set.
   */
  public static function name(): ?string {
    $id = ComposeContext::getServerId();
    return $id ? 'graphql_compose.settings.' . $id : NULL;
  }

  /**
   * Gets the server configuration for this context.
   *
   * @return \Drupal\Core\Config\ImmutableConfig|null
   *   The server configuration, or NULL if not set.
   */
  public static function config(): ?ImmutableConfig {
    static $config;
    $name = self::name();

    if ($name && !isset($config[$name])) {
      $config[$name] = \Drupal::configFactory()->get(self::name());
    }

    return $config[$name] ?? NULL;
  }

  /**
   * Gets a configuration value for this context.
   *
   * @param string $name
   *   The configuration name to get.
   * @param mixed $default
   *   The default value to return if the configuration is NULL.
   *
   * @return mixed
   *   The configuration value, or the default value if NULL.
   */
  public static function get(string $name, $default = NULL): mixed {
    $config = self::config();

    return $config
      ? ($config->get($name) ?? $default)
      : $default;
  }

}
