<?php

declare(strict_types=1);

namespace Drupal\graphql_compose\Utility;

/**
 * Utility for GraphQL Compose providers.
 *
 * Use during GraphQL runtime to get information on the enabled modules
 * for the current server context.
 *
 * If your module is not appearing, try add graphql_compose:graphql_compose
 * as a dependency in your .info.yml file.
 */
class ComposeProviders {

  /**
   * The default providers for the GraphQL Compose schema.
   *
   * @var string[]
   */
  public static $default = [
    'graphql_compose' => 'graphql_compose',
  ];

  /**
   * Gets the list of enabled providers for this context.
   *
   * This includes the default providers.
   *
   * @return string[]
   *   An array of provider module names that are enabled for this context.
   */
  public static function enabled(): array {
    $config = ComposeContext::getServerSchemaConfiguration();

    return array_merge(
      array_values(self::$default),
      array_values(array_filter($config['providers'] ?? []))
    );
  }

  /**
   * Checks if a provider is enabled for this context.
   *
   * @param string $name
   *   The provider name to check.
   *
   * @return bool
   *   TRUE if the provider is enabled, FALSE otherwise.
   */
  public static function has(string $name): bool {
    return in_array($name, self::enabled(), TRUE);
  }

  /**
   * Invokes the providers for this context.
   *
   * This executes hooks only for enabled providers, or hook that belong to
   * modules without a plugin for GraphQL Compose.
   *
   * @param string $hook
   *   The hook to invoke.
   * @param array $args
   *   The arguments to pass to the hook.
   */
  public static function invoke($hook, array $args = []) {
    $excluded = array_diff(
      array_keys(self::all()),
      self::enabled(),
    );

    /** @var \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler */
    $moduleHandler = \Drupal::service('module_handler');
    $moduleHandler->invokeAllWith($hook, function (callable $hook, string $module) use ($args, $excluded) {
      if (!in_array($module, $excluded, TRUE)) {
        $hook(...$args);
      }
    });
  }

  /**
   * Get the list of supported modules for GraphQL Compose.
   *
   * @return \Drupal\Core\Extension\Extension[]
   *   An array of module names that support GraphQL Compose.
   */
  public static function all(): array {
    static $modules = [];
    if (!empty($modules)) {
      return $modules;
    }

    // Find any schema extension plugins that use graphql_compose.
    $definitions = \Drupal::service('plugin.manager.graphql.schema_extension')->getDefinitions();
    foreach ($definitions as $definition) {
      if ($definition['schema'] === 'graphql_compose') {
        $modules[] = $definition['provider'];
      }
    }

    // Find any GraphQL Compose schema definitions.
    $definitions = \Drupal::service('graphql_compose.schema_type_manager')->getDefinitions();
    foreach ($definitions as $definition) {
      $modules[] = $definition['provider'];
    }

    // Find any GraphQL Compose entity definitions.
    $definitions = \Drupal::service('graphql_compose.entity_type_manager')->getDefinitions();
    foreach ($definitions as $definition) {
      $modules[] = $definition['provider'];
    }

    // Find any GraphQL Compose field definitions.
    $definitions = \Drupal::service('graphql_compose.field_type_manager')->getDefinitions();
    foreach ($definitions as $definition) {
      $modules[] = $definition['provider'];
    }

    // Remove duplicates.
    $modules = array_unique($modules);

    // Set the keys to the module names.
    $modules = array_combine($modules, $modules);

    // Sort providers by name.
    asort($modules);

    return $modules;
  }

}
