<?php

declare(strict_types=1);

namespace Drupal\Tests\graphql_compose\Functional\Contrib;

use Drupal\Tests\graphql_compose\Functional\GraphQLComposeBrowserTestBase;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\node\NodeInterface;

/**
 * Test the geomap_field Field integration.
 *
 * @group graphql_compose
 */
class GeomapFieldTest extends GraphQLComposeBrowserTestBase {

  /**
   * We aren't concerned with strict config schema for contrib.
   *
   * @var bool
   */
  protected $strictConfigSchema = FALSE; // @phpcs:ignore

  /**
   * The test node.
   *
   * @var \Drupal\node\NodeInterface
   */
  protected NodeInterface $node;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'geomap_field',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->createContentType([
      'type' => 'test',
      'name' => 'Test node type',
    ]);

    $field_storage = FieldStorageConfig::create([
      'field_name' => 'field_geo',
      'entity_type' => 'node',
      'type' => 'geomap',
    ]);

    $field_storage->save();

    $field = FieldConfig::create([
      'field_storage' => $field_storage,
      'bundle' => 'test',
      'label' => 'Geo',
    ]);

    $field->save();

    $this->node = $this->createNode([
      'type' => 'test',
      'title' => 'Test',
      'body' => [
        'value' => 'Test content',
        'format' => 'plain_text',
      ],
      'field_geo' => [
        'address_name' => 'Test Location',
        'street' => '123 Test St',
        'zipcode' => '12345',
        'city' => 'Test City',
        'country' => 'Test Country',
        'additional' => 'It is a test location.',
        'lat' => 12.345678,
        'lon' => 98.765432,
        'feature' => 'Test Feature',
      ],
      'status' => 1,
    ]);

    $this->setEntityConfig('node', 'test', [
      'enabled' => TRUE,
      'query_load_enabled' => TRUE,
    ]);

    $this->setFieldConfig('node', 'test', 'field_geo', [
      'enabled' => TRUE,
    ]);
  }

  /**
   * Test load entity by id.
   */
  public function testGeofield(): void {
    $query = <<<GQL
      query {
        node(id: "{$this->node->uuid()}") {
          ... on NodeTest {
            geo {
              addressName
              street
              zipcode
              city
              country
              additional
              lat
              lon
              feature
            }
          }
        }
      }
    GQL;

    $content = $this->executeQuery($query);

    $this->assertNotNull($content['data']['node']['geo'] ?? NULL);

    $geo = $content['data']['node']['geo'];

    $this->assertEquals('Test Location', $geo['addressName']);
    $this->assertEquals('123 Test St', $geo['street']);
    $this->assertEquals('12345', $geo['zipcode']);
    $this->assertEquals('Test City', $geo['city']);
    $this->assertEquals('Test Country', $geo['country']);
    $this->assertEquals('It is a test location.', $geo['additional']);
    $this->assertEquals(12.345678, $geo['lat']);
    $this->assertEquals(98.765432, $geo['lon']);
    $this->assertEquals('Test Feature', $geo['feature']);

  }

}
