<?php

declare(strict_types=1);

namespace Drupal\graphql_compose_mutations\Plugin\GraphQL\DataProducer;

use Drupal\Component\Utility\Xss;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityStorageException;
use Drupal\Core\Entity\EntityTypeBundleInfo;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Logger\LoggerChannelFactory;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\graphql\GraphQL\Execution\FieldContext;
use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;
use Drupal\graphql_compose_mutations\GraphQL\Response\GenericEntityResponse;
use Drupal\graphql_compose_mutations\Services\UserPermissions;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * This is a generic mutation class used by the SchemaExtension.
 *
 * @DataProducer(
 *   id = "generic_mutation_producer",
 *   name = @Translation("Mutation"),
 *   description = @Translation("Generic entity mutation extension."),
 *   produces = @ContextDefinition("any",
 *     label = @Translation("Any"),
 *   ),
 *   consumes = {
 *     "data" = @ContextDefinition("any",
 *       label = @Translation("Entity related data"),
 *     ),
 *     "metadata" = @ContextDefinition("any",
 *       label = @Translation("Generic mutation metadata"),
 *     ),
 *   },
 * )
 */
class GenericMutationProducer extends DataProducerPluginBase implements ContainerFactoryPluginInterface {

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected AccountInterface $currentUser;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The entity type bundle info.
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfo
   */
  protected EntityTypeBundleInfo $entityTypeBundleInfo;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected EntityFieldManagerInterface $entityFieldManager;

  /**
   * The Drupal logger factory service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactory
   */
  protected LoggerChannelFactory $loggerChannelFactory;


  /**
   * The ModuleHandlerInterface service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * The current module user_permissions service.
   *
   * @var \Drupal\graphql_compose_mutations\Services\UserPermissions
   */
  protected UserPermissions $userPermissions;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new self(
      $configuration,
      $plugin_id,
      $plugin_definition,
    );

    $instance->currentUser = $container->get('current_user');
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->entityTypeBundleInfo = $container->get('entity_type.bundle.info');
    $instance->entityFieldManager = $container->get('entity_field.manager');
    $instance->loggerChannelFactory = $container->get('logger.factory');
    $instance->moduleHandler = $container->get('module_handler');
    $instance->userPermissions = $container->get('graphql_compose_mutations.user_permissions');

    return $instance;
  }

  /**
   * A reusable generic mutation.
   *
   * @param array $data
   *   The mutation data.
   * @param array $metadata
   *   Several, entity-related metadata is required to create the mutation.
   * @param \Drupal\graphql\GraphQL\Execution\FieldContext $field_context
   *   The cache context.
   *
   * @return \Drupal\graphql_compose_mutations\GraphQL\Response\GenericEntityResponse|null
   *   The updated or new Entity or a GraphQL response.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function resolve(array $data, array $metadata, FieldContext $field_context): ?GenericEntityResponse {
    $response = new GenericEntityResponse();

    // node, group, message, profile etc.
    // @todo Limit allowed types from graphql_compose settings
    // This is already done on .graphqls files.
    $entity_type = strtolower($metadata["entity_type"]);
    $entity_bundle = strtolower($metadata["entity_bundle"]);
    $parent_entity_type = isset($metadata["parent_entity_type"]) ? strtolower($metadata["parent_entity_type"]) : NULL;
    $parent_entity_id = isset($metadata["parent_entity_id"]) ? (int) $metadata["parent_entity_id"] : NULL;
    $operation = strtolower($metadata["operation"]);
    if ($operation === "default") {
      $operation = "create";
    }

    $id = isset($metadata["id"]) ? (int) $metadata["id"] : NULL;
    // Default user. We can override this on the $data input argument.
    $user = $this->currentUser;
    $uid = $user->id();
    $entity = NULL;
    $custom_uid = NULL;
    $response->setSuccess(FALSE);
    $final_values = [];
    // @todo Check if there are other owner like fields on custom entity types.
    $user_ids = [
      "uid",
      "user_id",
    ];

    // Allow other modules to alter the initial data array.
    // See hook_graphql_compose_mutations_preprocess_data_alter.
    $this->moduleHandler->invokeAll('graphql_compose_mutations_preprocess_data_alter', [
      &$data,
      $entity_type,
      $entity_bundle,
      $operation,
    ]);

    // Add default uid to the data array and the metadata.
    if (in_array($operation, ["update", "delete"]) && $entity_type === "user" && !$id) {
      $id = $uid;
    }

    // Protect some entities from being deleted.
    if ($operation === "delete" && in_array($entity_type, ["user", "profile"]) && (int) $id === 1) {
      $protected_user = $this->t('User with uid @uid is protected', [
        "@uid" => $uid,
      ]);
      $response->addViolation($protected_user);
      return $response;
    }

    // Get parent entity (if we need it. E.g., for the votingapi entities).
    if ($parent_entity_type && $parent_entity_id) {
      $parent_entity = $this->entityTypeManager->getStorage($parent_entity_type)->load($parent_entity_id);
      if (!$parent_entity) {
        $missing_parent_entity = $this->t('Parent entity with id: @id and type: @type does not exist', [
          "@id" => $parent_entity_id,
          "@type" => $parent_entity_type,
        ]);
        $response->addViolation($missing_parent_entity);
        return $response;
      }

      // For some Mutations we can get some fields from metadata.
      if ($entity_type === "vote") {
        if (!isset($data["entity_type"]) && !isset($data["entity_id"])) {
          $data["entity_type"] = $parent_entity_type;
          $data["entity_id"] = $parent_entity_id;
        }

        if ($operation === "create" && !isset($data["value"])) {
          $data["value"] = 1;
        }
      }
    }

    // Get custom uid from data values.
    foreach ($user_ids as $user_id_field) {
      foreach ($data as $field => $value) {
        if ($field === $user_id_field) {
          $custom_uid = (int) $value;
        }
      }
    }

    if ($custom_uid && $custom_uid != $uid) {
      $uid = $custom_uid;

      // Load another User.
      $user = $this->entityTypeManager
        ->getStorage("user")
        ->load($uid);

      if (!$user) {
        $missing_user = $this->t('User with uid @uid does not exist', [
          "@uid" => $uid,
        ]);
        $response->addViolation($missing_user);
        return $response;
      }
    }

    $revision_log = $this->t("GraphQL entry. Operation: @operation, Type: @entity_type, Bundle: @entity_bundle, Uid: @uid", [
      '@operation' => $operation,
      '@entity_type' => $entity_type,
      '@entity_bundle' => $entity_bundle,
      '@uid' => $uid,
    ]);
    $final_values["revision_log"] = $revision_log;

    // Check entity bundle is valid.
    $entity_bundles = array_keys($this->entityTypeBundleInfo->getBundleInfo($entity_type));

    if (!$entity_bundle or !in_array($entity_bundle, $entity_bundles)) {
      $entity_bundles_error = $this->t('Entity bundle is required or is not correct. Please use one of @entity_bundles', [
        "@entity_bundles" => implode(", ", $entity_bundles),
      ]);
      $response->addViolation($entity_bundles_error);
      return $response;
    }

    // Check permissions early. We use a Service for that.
    // This also checks if there is an existing entity.
    $access = $this->userPermissions->userCanDoActionOnEntityByType(
      $operation,
      $user->id(),
      $id,
      $entity_type,
      $entity_bundle,
      $parent_entity_type,
      $parent_entity_id,
    );

    if ($access["access"] === FALSE) {
      $access_error = $access["error"] ?? $this->t('GenericMutationProducer: You do not have permission to perform this operation.');
      $response->addViolation($access_error);
      return $response;
    }

    if ($operation !== "create") {
      $entity = $this->entityTypeManager
        ->getStorage($entity_type)
        ->load($id);
    }

    // @todo Should we add the line below? Or we need an object as parameter?
    //   $field_context->addCacheableDependency($access);
    // Delete operation (does not need values) but may have errors.
    if ($operation === "delete") {
      try {
        $delete_status = $entity->delete();
      }
      catch (\Exception $e) {
        $message = $e->getMessage();
        $response->addViolation($message);
        return $response;
      }

      if ($delete_status instanceof EntityStorageException) {
        $delete_error = $delete_status->getMessage();
        $response->addViolation($delete_error);
      }
      else {
        $response->setSuccess(TRUE);
      }

      $response->setGenericEntity(NULL);
      return $response;
    }

    // For non delete operations check additional values exist on input.
    if (empty($data)) {
      // If we have no entity values, throw an error.
      $missing_data = $this->t('Missing data values.');
      $response->addViolation($missing_data);
      return $response;
    }

    // Final fields mapping
    // Get all the entity fields for this bundle.
    $entity_fields = [];
    $read_only_fields_to_keep = [
      "system_entity_is_from_api",
    ];
    $entity_fields_all = $this->entityFieldManager->getFieldDefinitions($entity_type, $entity_bundle);
    // Remove computed fields from graphql_compose_* modules.
    foreach ($entity_fields_all as $field_name => $field) {
      $field_array = $field->toArray();
      $provider = $field_array["provider"] ?? "";
      $is_computed = $field_array["computed"] ?? FALSE;
      $is_graphql_compose = $provider && str_starts_with($provider, "graphql_compose");
      if (!($is_computed && $is_graphql_compose)) {
        $entity_fields[$field_name] = $field;
      }
      // Exceptions for some special read-only fields.
      if (in_array($field_name, $read_only_fields_to_keep)) {
        $entity_fields[$field_name] = $field;
      }
    }
    $entity_fields_keys = array_keys($entity_fields);

    // Check if $data contains "bad" fields and throw an error.
    foreach ($data as $field => $value) {
      if (!in_array($field, $entity_fields_keys)) {
        $bad_field = $this->t('Field "@field" is not valid key for data.', [
          "@field" => $field,
        ]);
        $response->addViolation($bad_field);
        return $response;
      }
    }

    // @todo Validate complex or composite values (e.g. entity_reference)
    // @todo Find a way to save the mapping to a config yml
    // E.g. the body field needs ["value" => "", "format" => "basic_html"].
    // @codingStandards
    foreach ($entity_fields_keys as $machine_name) {
      // Do not process computed fields.
      foreach ($data as $field => $value) {
        if ($field === $machine_name) {
          if (!is_array($value) && $value !== "") {
            $value = Xss::filter($value);
          }
          $final_values[$machine_name] = $value;
        }
      }
    }
    // Special treatment with the uid. Some entity types have another field.
    if (in_array("uid", $entity_fields_keys) && $operation === "create") {
      $final_values["uid"] = $uid;
    }
    if (in_array("user_id", $entity_fields_keys)) {
      $final_values["user_id"] = $uid;
    }
    else {
      unset($final_values["user_id"]);
    }

    // Clean up manual added values.
    if (!isset($entity_fields["revision_log"])) {
      unset($final_values["revision_log"]);
    }

    // Operation create.
    if ($operation === "create") {

      // Some entity types do not use "type" as the bundle machine name.
      // We need to get this through the entityTypeManager.
      $storage = $this->entityTypeManager->getStorage($entity_type);
      $type_keys = $storage->getEntityType()->getKeys();
      $type_machine_name = $type_keys["bundle"] ?? "";

      // Finally, just create the new entity.
      $entity = $storage->create([$type_machine_name => $entity_bundle]);
    }

    // Allow other modules to alter the final values.
    // See hook_graphql_compose_mutations_preprocess_values_alter.
    $this->moduleHandler->invokeAll('graphql_compose_mutations_preprocess_values_alter', [
      &$final_values,
      $entity_type,
      $entity_bundle,
      $operation,
    ]);

    // Add extra values and save the entity.
    foreach ($final_values as $machine_name => $value) {
      try {
        if (method_exists($entity, 'set')) {
          $entity->set($machine_name, $value);
        }
      }
      catch (\Exception $e) {
        $message = $e->getMessage();
        $response->addViolation($message);
        return $response;
      }
    }

    // @todo Validate entity (this needs the dedicated Entity class usage)
    try {
      $save_status = $entity->save();
    }
    catch (\Exception $e) {
      $message = $e->getMessage();
      $response->addViolation($message);
      return $response;
    }

    // If an error occurred (wrong values etc) we do not get Int values.
    // Type "private_message_thread" returns NULL on UPDATE action.
    // @todo May need to provide a patch on the module private_message itself.
    if ($entity_type === "private_message_thread") {
      if ($operation !== "create") {
        $entity = $this->entityTypeManager
          ->getStorage($entity_type)
          ->load($id);
        if (!$entity) {
          $save_error = "Error on entity Save for " . $revision_log;
          $this->loggerChannelFactory->get("graphql_compose_mutations")
            ->error($save_error);
          $response->addViolation($save_error);
          return $response;
        }
      }
      else {
        if ($entity->id()) {
          // Manual set $save_status to 1 for the new private_message_thread.
          $save_status = 1;
        }
      }
    }
    elseif ($save_status !== 1 && $save_status !== 2) {
      // 1: SAVED_NEW, 2: SAVED_UPDATED.
      $save_error = "Wrong Save status number: " . $save_status . " on entity Save for " . $revision_log;
      $this->loggerChannelFactory->get("graphql_compose_mutations")->error($save_error);
      $response->addViolation($save_error);
      return $response;
    }

    // @todo Should we add cache context for the entity?
    //   $field_context->addCacheableDependency($entity);
    // Pass results to the response.
    $response->setGenericEntity($entity);
    if (empty($response->getViolations())) {
      $response->setSuccess(TRUE);
    }

    return $response;
  }

}
