<?php

declare(strict_types=1);

namespace Drupal\graphql_compose_mutations\Plugin\GraphQL\DataProducer;

use Drupal\Core\Entity\EntityTypeBundleInfo;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;
use Drupal\graphql_compose_mutations\GraphQL\Response\OperationsByEntityTypeResponse;
use Drupal\graphql_compose_mutations\Services\UserPermissions;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * This is a helper class that returns the allowed operations by entity type.
 *
 * @DataProducer(
 *   id = "operations_by_entity_type_producer",
 *   name = @Translation("Operations by Entity Type"),
 *   description = @Translation("a helper class that returns allowed operations by entity type."),
 *   produces = @ContextDefinition("any",
 *     label = @Translation("Any"),
 *   ),
 *   consumes = {
 *     "entity_type" = @ContextDefinition("any",
 *       label = @Translation("Entity type"),
 *     ),
 *   },
 * )
 */
class OperationsByEntityTypeProducer extends DataProducerPluginBase implements ContainerFactoryPluginInterface {

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected AccountInterface $currentUser;

  /**
   * The entity type bundle info.
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfo
   */
  protected EntityTypeBundleInfo $entityTypeBundleInfo;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The current module user_permissions service.
   *
   * @var \Drupal\graphql_compose_mutations\Services\UserPermissions
   */
  protected UserPermissions $userPermissions;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new self(
      $configuration,
      $plugin_id,
      $plugin_definition,
    );

    $instance->currentUser = $container->get('current_user');
    $instance->entityTypeBundleInfo = $container->get('entity_type.bundle.info');
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->userPermissions = $container->get('graphql_compose_mutations.user_permissions');

    return $instance;
  }

  /**
   * A query to return allowed operations by entity type.
   *
   * @param string $entity_type
   *   The entity_type.
   *
   * @return \Drupal\graphql_compose_mutations\GraphQL\Response\OperationsByEntityTypeResponse|null
   *   The OperationsByEntityType response.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   */
  public function resolve(string $entity_type): ?OperationsByEntityTypeResponse {
    $response = new OperationsByEntityTypeResponse();
    $results = [];
    $user = $this->currentUser;
    $uid = $user->id();
    $entity_type = strtolower($entity_type);
    // Some entity types do not have bundles. So they have the type as bundle.
    $bundles = $this->entityTypeBundleInfo->getBundleInfo($entity_type) ?? [$entity_type];
    // @todo add more operations if needed.
    $operations = [
      "create",
    ];

    if (!$entity_type) {
      $entity_type_error = $this->t('Entity type is required.');
      $response->addViolation($entity_type_error);
      return $response;
    }

    foreach ($bundles as $bundle_key => $bundle_values) {
      $bundle_entity_type = $this->entityTypeManager->getDefinition($entity_type)->getBundleEntityType();
      $bundle_description = "";
      $bundle_label = $bundle_key;

      if ($bundle_entity_type) {
        $bundle_definition = $this->entityTypeManager
          ->getStorage($bundle_entity_type)
          ->load($bundle_key);
        $bundle_description = $bundle_definition->get("description");
        $bundle_label = $bundle_definition->label();
      }

      foreach ($operations as $operation) {
        $label = $operation . " " . $bundle_label;
        if ($operation === "create") {
          $label = "New " . $bundle_label;
        }
        // Important. The "create" operation does not need an entity_id.
        // But we should make it more generic for other operations.
        // In same cases it needs the Parent entity.
        $access = $this->userPermissions->userCanDoActionOnEntityByType(
          $operation,
          $uid,
          "",
          $entity_type,
          (string) $bundle_key,
          "",
          ""
        );
        $results[$entity_type][$bundle_key][$operation] = [
          "access" => $access["access"],
          "label" => $label,
          "description" => $bundle_description,
        ];
      }
    }

    if (empty($response->getViolations())) {
      $response->setOperations($results);
    }

    return $response;
  }

}
