<?php

declare(strict_types=1);

namespace Drupal\graphql_compose_mutations\Plugin\GraphQL\DataProducer;

use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactory;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;
use Drupal\graphql_compose_mutations\GraphQL\Response\PermissionsResponse;
use Drupal\graphql_compose_mutations\Services\UserPermissions;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * This is a helper class that checks for User operations.
 *
 * @DataProducer(
 *   id = "permissions_producer",
 *   name = @Translation("Permissions"),
 *   description = @Translation("a helper class that checks for user operations."),
 *   produces = @ContextDefinition("any",
 *     label = @Translation("Any"),
 *   ),
 *   consumes = {
 *     "operation" = @ContextDefinition("any",
 *       label = @Translation("Operation"),
 *     ),
 *     "uid" = @ContextDefinition("any",
 *       label = @Translation("The User uid"),
 *     ),
 *     "entity_id" = @ContextDefinition("any",
 *       label = @Translation("Entity ID"),
 *       required = FALSE,
 *     ),
 *     "entity_type" = @ContextDefinition("any",
 *       label = @Translation("Entity type"),
 *     ),
 *     "entity_bundle" = @ContextDefinition("any",
 *       label = @Translation("Entity bundle"),
 *     ),
 *     "parent_entity_type" = @ContextDefinition("any",
 *       label = @Translation("Parent Entity type (optional)"),
 *       default_value = "",
 *       required = FALSE,
 *     ),
 *     "parent_entity_id" = @ContextDefinition("any",
 *       label = @Translation("Parent Entity ID (optional)"),
 *       default_value = "",
 *       required = FALSE,
 *     ),
 *   },
 * )
 */
class PermissionsProducer extends DataProducerPluginBase implements ContainerFactoryPluginInterface {

  /**
   * The current module user_permissions service.
   *
   * @var \Drupal\graphql_compose_mutations\Services\UserPermissions
   */
  protected UserPermissions $userPermissions;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The Drupal logger factory service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactory
   */
  protected LoggerChannelFactory $loggerChannelFactory;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new self(
      $configuration,
      $plugin_id,
      $plugin_definition,
    );

    $instance->userPermissions = $container->get('graphql_compose_mutations.user_permissions');
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->loggerChannelFactory = $container->get('logger.factory');

    return $instance;
  }

  /**
   * A query to check for operation permissions by uid.
   *
   * @return \Drupal\graphql_compose_mutations\GraphQL\Response\PermissionsResponse|null
   *   The access Permissions response.
   */
  public function resolve(
    string $operation,
    mixed $uid,
    mixed $entity_id,
    string $entity_type,
    string $entity_bundle,
    mixed $parent_entity_type,
    mixed $parent_entity_id,
  ): ?PermissionsResponse {
    $response = new PermissionsResponse();
    $response->setAccess(FALSE);

    $entity_type = strtolower($entity_type);
    $entity_bundle = strtolower($entity_bundle);
    $operation = strtolower($operation);

    $parent_entity_type = $parent_entity_type ? strtolower($parent_entity_type) : NULL;
    $parent_entity_id = $parent_entity_id ? (int) $parent_entity_id : NULL;

    if ($parent_entity_type && $parent_entity_id) {
      try {
        $parent = $this->entityTypeManager->getStorage($parent_entity_type)
          ->load($parent_entity_id);
        if (!$parent) {
          $missing_parent_entity = $this->t('Parent entity with id @uid and type @type does not exist', [
            "@id" => $parent_entity_id,
            "@type" => $parent_entity_type,
          ]);
          $response->addViolation($missing_parent_entity);
          return $response;
        }
      }
      catch (InvalidPluginDefinitionException | PluginNotFoundException $e) {
        $message = $e->getMessage();
        $this->loggerChannelFactory->get("graphql_compose_mutations")->error($message);
        $response->addViolation($message);
        return $response;
      }
    }

    // Get access array from Service.
    try {
      $access = $this->userPermissions->userCanDoActionOnEntityByType(
        $operation,
        $uid,
        $entity_id,
        $entity_type,
        $entity_bundle,
        $parent_entity_type,
        $parent_entity_id,
      );

      // Disallowed.
      if ($access["access"] === FALSE && isset($access["error"])) {
        $access_error = $access["error"];
        $response->addViolation($access_error);
        return $response;
      }

      $response->setAccess($access["access"]);
    }
    catch (InvalidPluginDefinitionException | PluginNotFoundException $e) {
      $message = $e->getMessage();
      $this->loggerChannelFactory->get("graphql_compose_mutations")->error($message);
      $response->addViolation($message);
      return $response;
    }

    return $response;
  }

}
