<?php

namespace Drupal\graphql_compose_mutations\Services;

use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Entity\EntityTypeBundleInfo;
use Drupal\Core\Entity\EntityTypeManager;
use Drupal\Core\Extension\ModuleHandler;
use Drupal\Core\Logger\LoggerChannel;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\node\NodeInterface;
use Drupal\user\UserInterface;

/**
 * A helper service to get User permissions by operation.
 */
class UserPermissions {
  /**
   * The User operations allowed to check for.
   *
   * @var array
   *
   * @todo This is a duplicate of enum OperationExtended at generic_mutation.base.graphqls
   * @todo Maybe we could use the existing GraphQL data producer "entity_access".
   * @todo Add more non-core operation verbs: "like, dislike, flag, unflag, enroll, comment, message etc"
   */
  public array $operations = [
    // Core operations.
    "create",
    "delete",
    "update",
    "view",
  ];

  /**
   * The ModuleHandler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandler
   */
  protected ModuleHandler $moduleHandler;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected AccountInterface $currentUser;

  /**
   * Drupal\Core\Entity\EntityTypeManager definition.
   *
   * @var \Drupal\Core\Entity\EntityTypeManager
   */
  protected EntityTypeManager $entityTypeManager;

  /**
   * Drupal\Core\Entity\EntityTypeBundleInfo definition.
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfo
   */
  protected EntityTypeBundleInfo $entityTypeBundleInfo;

  /**
   * The logger service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $logger;

  /**
   * The Service constructor.
   *
   * @param \Drupal\Core\Extension\ModuleHandler $moduleHandler
   *   The ModuleHandler service.
   * @param \Drupal\Core\Session\AccountInterface $currentUser
   *   The current user.
   * @param \Drupal\Core\Entity\EntityTypeManager $entityTypeManager
   *   The entity_type.manager service.
   * @param \Drupal\Core\Logger\LoggerChannel $logger
   *   The logger.channel.graphql service.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfo $entityTypeBundleInfo
   *   The entity_type.manager service.
   */
  public function __construct(
    ModuleHandler $moduleHandler,
    AccountInterface $currentUser,
    EntityTypeManager $entityTypeManager,
    LoggerChannel $logger,
    EntityTypeBundleInfo $entityTypeBundleInfo,
  ) {
    // Get required services.
    $this->moduleHandler = $moduleHandler;
    $this->currentUser = $currentUser;
    $this->entityTypeManager = $entityTypeManager;
    $this->logger = $logger;
    $this->entityTypeBundleInfo = $entityTypeBundleInfo;
  }

  /**
   * Checks if an entity type (e.g. node, taxonomy_term) exists.
   *
   * @param string $entity_type
   *   The entity type to check for.
   *
   * @return bool
   *   Return if entity type exists.
   */
  private function entityTypeExists(string $entity_type):bool {
    $entity_types = $this->entityTypeBundleInfo->getAllBundleInfo();
    $entity_type_keys = array_keys($entity_types);

    return in_array($entity_type, $entity_type_keys);
  }

  /**
   * Checks if an entity bundle (e.g. event, page of type Node) exists.
   *
   * @param string $entity_type
   *   The entity type to check for.
   * @param mixed $entity_bundle
   *   The entity bundle to check for.
   *
   * @return bool
   *   Return if entity bundle of this type exists.
   */
  private function entityBundleOfTypeExists(string $entity_type, mixed $entity_bundle):bool {
    if (!$entity_bundle) {
      $entity_bundle = $entity_type;
    }

    $entity_bundles = $this->entityTypeBundleInfo->getBundleInfo($entity_type);
    $entity_bundles_keys = array_keys($entity_bundles);

    return in_array($entity_bundle, $entity_bundles_keys);
  }

  /**
   * Get generic permissions manually added. This is a helper method.
   *
   * @param string $entity_bundle
   *   The entity bundle to get the generic permissions for.
   *
   * @return array
   *   Array of manually added permissions.
   */
  private function getGenericPermissionsByEntityBundle(string $entity_bundle): array {
    // Unfortunately, modules do not use a pattern for permissions.
    // We have to generate this list manually...
    // @todo only add permissions if entity types exist.
    $permissions = [
      "comment" => [
        "administer comments",
        "post comments",
        "create " . $entity_bundle . " comment",
      ],
      "contact_message" => [
        "administer contact forms",
        "access site-wide contact form",
      ],
      "message" => [
        // @todo This is a system entity so there are no other permissions.
        "administer messages",
      ],
      "node" => [
        "administer nodes",
        "bypass node access",
        "create " . $entity_bundle . " content",
      ],
      "group" => [
        "administer group",
        "bypass create group access",
        "bypass group access",
        "manage all groups",
        "create " . $entity_bundle . " group",
      ],
      "private_message" => [
        "administer private messages",
        "use private messaging system",
      ],
      "profile" => [
        "administer profile",
        "create " . $entity_bundle . " profile",
      ],
      "taxonomy_term" => [
        "administer taxonomy",
        "create terms in " . $entity_bundle,
      ],
    ];

    // Module flag.
    if ($this->moduleHandler->moduleExists('flag')) {
      $permissions['flagging'] = [
        "flag " . $entity_bundle,
        "unflag " . $entity_bundle,
      ];
    }

    // Module contact_storage.
    if ($this->moduleHandler->moduleExists('contact_storage')) {
      $permissions['contact_storage'] = [
        "administer contact forms",
        "access site-wide contact form",
      ];
    }

    // Open Social additional conditions.
    if ($this->moduleHandler->moduleExists('activity_creator')) {
      $permissions['activity'] = [
        "administer activity entities",
        "add activity entities",
      ];
    }

    if ($this->moduleHandler->moduleExists('social_event')) {
      $permissions['event_enrollment'] = [
        "administer event enrollment entities",
        "add event enrollment entities",
      ];
    }

    if ($this->moduleHandler->moduleExists('social_private_message')) {
      $permissions['private_message'][] = "create private messages thread";
    }

    if ($this->moduleHandler->moduleExists('social_post')) {
      $permissions['post'] = [
        "administer post entities",
        "add post entities",
        "add " . $entity_bundle . " post entities",
      ];
    }

    ksort($permissions);

    return $permissions;
  }

  /**
   * Load user from uid. We use the entityTypeManager for dependency injection.
   *
   * @param mixed $uid
   *   The uid.
   *
   * @return \Drupal\user\UserInterface|null
   *   The final User object.
   */
  private function loadAccountFromUid(mixed $uid): ?UserInterface {
    try {
      $user_storage = $this->entityTypeManager->getStorage("user");
      $account = $user_storage->load($uid);
      if ($account instanceof UserInterface) {
        return $account;
      }
    }
    catch (InvalidPluginDefinitionException | PluginNotFoundException $e) {
      $this->logger->error($e->getMessage());
    }
    return NULL;
  }

  /**
   * Check if a user (loaded from uid) has a permission to CREATE.
   *
   * @param mixed $uid
   *   The uid.
   * @param string $entity_type
   *   The entity type (node, post, taxonomy_term, etc)
   * @param mixed $entity_bundle
   *   The entity bundle.
   * @param mixed $parent_entity_type
   *   The parent entity type that may be related to this operation. E.g. for
   *   vote entity types this may be the Node or Comment we are voting for.
   * @param mixed $parent_entity_id
   *   The parent entity ID. See above.
   *
   * @return bool
   *   Return final access.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function userCanCreateEntityByType(
    mixed $uid,
    string $entity_type,
    mixed $entity_bundle,
    mixed $parent_entity_type,
    mixed $parent_entity_id,
  ):bool {
    $final_access = FALSE;
    $account = $this->loadAccountFromUid($uid);

    if ($entity_type === "group_content") {
      if (!$parent_entity_type || !$parent_entity_id) {
        return FALSE;
      }
      $parent_entity = $this->entityTypeManager->getStorage($parent_entity_type)->load($parent_entity_id);
      // phpcs:disable
      if ($parent_entity instanceof \Drupal\group\Entity\GroupInterface) {
        // phpcs:enable
        /** @var \Drupal\group\Entity\GroupInterface $parent_entity */
        // If this is a try to CREATE a new Group Membership.
        if (str_contains($entity_bundle, "-group_membership")) {
          // @todo Check field allow_request value also.
          $owner_id = $parent_entity->getOwner()->id();
          // This is the owner already.
          if ((int) $owner_id === (int) $uid) {
            return FALSE;
          }

          $group_content = $this->entityTypeManager
            ->getStorage("group_content")
            ->loadByProperties([
              "type" => $entity_bundle,
              "gid" => $parent_entity_id,
              "entity_id" => $uid,
            ]);

          // There is already a Group Membership. Skip.
          if ($group_content) {
            return FALSE;
          }

          return TRUE;
        }

        // @todo What are the Permissions by Group for the other bundles?
        // @todo Maybe there is a more dedicated Service to handle this.
        return TRUE;
      }
      else {
        return FALSE;
      }
    }

    if ($entity_type === "event_enrollment") {
      if ($this->moduleHandler->moduleExists('social_event')) {
        if (!$parent_entity_type || !$parent_entity_id) {
          return FALSE;
        }
        $parent_entity = $this->entityTypeManager->getStorage($parent_entity_type)->load($parent_entity_id);
        if (!$parent_entity instanceof NodeInterface) {
          return FALSE;
        }
        else {
          /** @var \Drupal\social_event\Entity\Node\Event $parent_entity */
          $event_is_enroll_enabled = $parent_entity->isEnrollmentEnabled();
          if (!$event_is_enroll_enabled) {
            return FALSE;
          }

          $event_finished = $this->socialEventHasBeenFinished($parent_entity);
          if ($event_finished) {
            return FALSE;
          }

          // Get enrollment for this Parent entity and this User.
          $enrollment = $this->entityTypeManager
            ->getStorage("event_enrollment")
            ->loadByProperties([
              "field_event" => $parent_entity_id,
              "field_account" => $uid,
            ]);
          // Already enrolled to the Event.
          if ($enrollment) {
            return FALSE;
          }
        }
      }
    }

    if ($entity_type === "vote") {
      if (!$parent_entity_type || !$parent_entity_id) {
        return FALSE;
      }
      $parent_entity = $this->entityTypeManager->getStorage($parent_entity_type)->load($parent_entity_id);
      return like_and_dislike_can_vote($account, $entity_bundle, $parent_entity);
    }

    if ($entity_type === "flagging") {
      if (!$parent_entity_type || !$parent_entity_id) {
        return FALSE;
      }
      $parent_entity = $this->entityTypeManager->getStorage($parent_entity_type)->load($parent_entity_id);
      if (!$parent_entity) {
        return FALSE;
      }
      /** @var \Drupal\Core\Entity\EntityInterface $parent_entity */
      $flagging_entity = $this->entityTypeManager->getStorage($entity_type)
        ->loadByProperties([
          "flag_id" => $entity_bundle,
          "uid" => $uid,
          "entity_type" => $parent_entity_type,
          "entity_id" => $parent_entity_id,
        ]);

      if ($flagging_entity) {
        return FALSE;
      }
    }

    if ((int) $uid === 1) {
      return TRUE;
    }

    $roles = $account->getRoles();
    foreach ($roles as $role_id) {
      /** @var \Drupal\user\Entity\Role $role */
      $role = $this->entityTypeManager->getStorage("user_role")->load($role_id);
      if ($role->isAdmin()) {
        return TRUE;
      }
    }

    // Core entity access calculations.
    $access_handler = $this->entityTypeManager->getAccessControlHandler($entity_type);
    $access_result = (bool) $access_handler->createAccess($entity_bundle, $account);
    if ($access_result) {
      return TRUE;
    }

    // Final, manual added access checks.
    $create_permissions = $this->getGenericPermissionsByEntityBundle($entity_bundle);
    if (isset($create_permissions[$entity_type])) {
      foreach ($create_permissions[$entity_type] as $permission) {
        $access = $account->hasPermission($permission);
        if ($access === TRUE) {
          $final_access = TRUE;
        }
      }
    }

    return $final_access;
  }

  /**
   * Check if a user (loaded from uid) has a permission to perform CRUD actions.
   *
   * @param string $operation
   *   The operation to check for,.
   * @param mixed $uid
   *   The uid.
   * @param mixed $entity_id
   *   The entity id to load.
   * @param string $entity_type
   *   The entity type (node, post, taxonomy_term etc.)
   * @param mixed $entity_bundle
   *   The entity bundle.
   * @param mixed $parent_entity_type
   *   The parent entity type that may be related to this operation. E.g. for
   *   vote entity types this may be the Node or Comment we are voting for.
   * @param mixed $parent_entity_id
   *   The parent entity ID. See above.
   *
   * @return array
   *   Return final access array with useful info.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function userCanDoActionOnEntityByType(
    string $operation,
    mixed $uid,
    mixed $entity_id,
    string $entity_type,
    mixed $entity_bundle,
    mixed $parent_entity_type,
    mixed $parent_entity_id,
  ):array {
    $result = [];
    $current_user = $this->currentUser;
    $final_access = FALSE;
    $account = $this->loadAccountFromUid($uid);
    $operations = $this->operations;
    $entity_type_exists = $this->entityTypeExists($entity_type);
    $entity_bundle_exists = $this->entityBundleOfTypeExists($entity_type, $entity_bundle);

    if (!in_array($operation, $operations)) {
      return [
        "access" => FALSE,
        "error" => "userCanDoActionOnEntityByType. Operation is not allowed.",
      ];
    }

    if (!$account) {
      return [
        "access" => FALSE,
        "error" => "userCanDoActionOnEntityByType. Account for uid " . $uid . " does not exist.",
      ];
    }

    if (!$entity_type_exists) {
      return [
        "access" => FALSE,
        "error" => "userCanDoActionOnEntityByType. Entity type" . $entity_type . " does not exist.",
      ];
    }

    if (!$entity_bundle_exists) {
      return [
        "access" => FALSE,
        "error" => "userCanDoActionOnEntityByType. Entity bundle " . $entity_bundle . " of type " . $entity_type . " does not exist.",
      ];
    }

    // Allow other modules to alter permissions at the beginning.
    // See hook_graphql_compose_mutations_preprocess_permissions_alter().
    $this->moduleHandler->invokeAll('graphql_compose_mutations_preprocess_permissions_alter', [
      &$result,
      $operation,
      $entity_type,
      $uid,
      $entity_bundle,
      $entity_id,
      $parent_entity_type,
      $parent_entity_id,
    ]);

    if (isset($result['access']) && $result['access'] === FALSE) {
      return $result;
    }

    if ($operation === "create") {
      $types_with_parent = [
        "event_enrollment",
        "flagging",
        "group_content",
        "vote",
      ];

      if (in_array($entity_type, $types_with_parent) && (!$parent_entity_type || !$parent_entity_id)) {
        return [
          "access" => FALSE,
          "error" => "userCanDoActionOnEntityByType. Values for parent_entity_type and parent_entity_id fields are required for type: " . $entity_type,
        ];
      }

      $access = $this->userCanCreateEntityByType($uid, $entity_type, $entity_bundle, $parent_entity_type, $parent_entity_id);
      $result = [
        "access" => $access,
      ];

      if (!$access) {
        $result["error"] = "userCanDoActionOnEntityByType: Access denied for uid " . $uid;
      }
      return $result;
    }

    // Load existing entity.
    try {
      $entity = $this->entityTypeManager
        ->getStorage($entity_type)
        ->load($entity_id);
      if ($entity) {
        if ($entity_type === "group_content") {
          if ($operation === "delete") {
            /** @var \Drupal\group\Entity\GroupContent $entity */
            $gc_id = $entity->get("entity_id")->getString();
            if ((int) $gc_id === (int) $uid) {
              return [
                "access" => TRUE,
              ];
            }
          }
        }

        if ($entity_type === "flagging") {
          if ($operation === "delete") {
            /** @var \Drupal\flag\Entity\Flagging $entity */
            $flagging_uid = $entity->getOwner()->id();
            if ((int) $flagging_uid === (int) $uid) {
              return [
                "access" => TRUE,
              ];
            }
          }
        }

        if ($entity_type === "event_enrollment") {
          if ($operation === "delete") {
            /** @var \Drupal\social_event\Entity\EventEnrollment $entity */
            $enrollment_uid = $entity->getOwner()->id();
            if ((int) $enrollment_uid === (int) $uid) {
              return [
                "access" => TRUE,
              ];
            }
          }
        }

        // Special override for the votingapi module. Check access early.
        if ($entity_type === "vote") {
          if (!$parent_entity_type || !$parent_entity_id) {
            return [
              "access" => FALSE,
              "error" => "userCanDoActionOnEntityByType. Parent entity is required for bundle: " . $entity_bundle . " of type: " . $entity_type,
            ];
          }
          $parent_entity = $this->entityTypeManager->getStorage($parent_entity_type)->load($parent_entity_id);
          if ($parent_entity) {
            if ($operation === "delete") {
              /** @var \Drupal\votingapi\Entity\Vote $entity */
              $vote_uid = $entity->get("user_id")->getString();
              if ($vote_uid !== $uid) {
                return [
                  "access" => FALSE,
                  "error" => "userCanDoActionOnEntityByType. Vote entity with id: " . $entity_id . " is not from this User.",
                ];
              }
            }
            $final_access = like_and_dislike_can_vote($account, $entity_bundle, $parent_entity);
          }
        }

        // private_message_thread.
        if ($entity_type === "private_message_thread") {
          /** @var \Drupal\private_message\Entity\PrivateMessageThread $entity */
          $members = $entity->getMembersId();
          if (in_array($current_user->id(), $members)) {
            $final_access = TRUE;
          }

          if ($current_user->hasPermission("use private messaging system")) {
            $final_access = TRUE;
          }
        }

        // Initially try to get access and exit with success.
        if (!$final_access) {
          $final_access = $entity->access($operation, $account);
        }

        // Allow other modules to alter permissions at the end.
        // See hook_graphql_compose_mutations_postprocess_permissions_alter.
        $this->moduleHandler->invokeAll('graphql_compose_mutations_postprocess_permissions_alter', [
          &$result,
          $operation,
          $entity_type,
          $uid,
          $entity_bundle,
          $entity_id,
          $parent_entity_type,
          $parent_entity_id,
        ]);

        if (isset($result['access']) && $result['access'] === FALSE) {
          return $result;
        }

        if ($final_access === TRUE) {
          return [
            "access" => TRUE,
          ];
        }

        $result = [
          "access" => $final_access,
        ];

        if (!$final_access && !isset($result["error"])) {
          $result["error"] = "userCanDoActionOnEntityByType. Entity exists. Access denied for uid " . $uid;
        }
        return $result;
      }
      else {
        $entity_error = 'userCanDoActionOnEntityByType. Entity of type: ' . $entity_type . ', bundle: ' . $entity_bundle . ' with ID: ' . $entity_id . ' does not exist.';
        $this->logger->warning($entity_error);
        return [
          "access" => FALSE,
          "error" => $entity_error,
        ];
      }
    }
    catch (InvalidPluginDefinitionException | PluginNotFoundException $e) {
      $this->logger->error($e->getMessage());
      return [
        "access" => FALSE,
        "error" => "userCanDoActionOnEntityByType | " . $e->getMessage(),
      ];
    }
  }

  /**
   * Function to determine if an event has been finished.
   *
   * This is a pure copy from Class social_event/src/SocialEventTrait.php
   * Unfortunately, we can use a private method from a Trait.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The event to check for.
   *
   * @return bool
   *   TRUE if the evens is finished/completed.
   */
  public function socialEventHasBeenFinished(NodeInterface $node): bool {
    $current_time = new DrupalDateTime();

    /** @var \Drupal\Core\Datetime\DrupalDateTime $check_end_date */
    $check_end_date = $node->get('field_event_date_end')->isEmpty()
      // @phpstan-ignore-next-line
      ? $node->get('field_event_date')->date
      // @phpstan-ignore-next-line
      : $node->get('field_event_date_end')->date;

    if (!$check_end_date instanceof DrupalDateTime) {
      return FALSE;
    }

    $check_all_day = !$node->get('field_event_all_day')->isEmpty()
      ? $node->get('field_event_all_day')->getString()
      : NULL;

    return $current_time->getTimestamp() > $check_end_date->getTimestamp() &&
      !($check_all_day && $check_end_date->format('Y-m-d') === $current_time->format('Y-m-d'));
  }

}
